/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database;

import com.intellij.database.model.DatabaseSystem;
import com.intellij.database.model.RawConnectionConfig;
import com.intellij.openapi.util.text.StringUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * @author gregsh
 */
public final class DatabaseFamilyId {

  public static final DatabaseFamilyId UNKNOWN = new DatabaseFamilyId("UNKNOWN");
  public static final DatabaseFamilyId ORACLE = new DatabaseFamilyId("ORACLE");
  public static final DatabaseFamilyId MYSQL = new DatabaseFamilyId("MYSQL");
  public static final DatabaseFamilyId POSTGRES = new DatabaseFamilyId("POSTGRES");
  public static final DatabaseFamilyId MSSQL = new DatabaseFamilyId("MSSQL");
  public static final DatabaseFamilyId SYBASE = new DatabaseFamilyId("SYBASE");
  public static final DatabaseFamilyId DB2 = new DatabaseFamilyId("DB2");
  public static final DatabaseFamilyId SQLITE = new DatabaseFamilyId("SQLITE");
  public static final DatabaseFamilyId HSQLDB = new DatabaseFamilyId("HSQLDB");
  public static final DatabaseFamilyId H2 = new DatabaseFamilyId("H2");
  public static final DatabaseFamilyId DERBY = new DatabaseFamilyId("DERBY");

  private final String myName;

  public DatabaseFamilyId(@NotNull  String name) {
    myName = name;
  }

  @Override
  public String toString() {
    return myName;
  }

  public boolean isOracle() { return this == ORACLE; }
  public boolean isMysql() { return this == MYSQL; }
  public boolean isPostgres() { return this == POSTGRES; }
  public boolean isMicrosoft() { return this == MSSQL; }
  public boolean isSybase() { return this == SYBASE; }
  public boolean isDb2() { return this == DB2; }
  public boolean isHsqldb() { return this == HSQLDB; }
  public boolean isH2() { return this == H2; }
  public boolean isDerby() { return this == DERBY; }
  public boolean isSqlite() { return this == SQLITE; }
  public boolean isTransactSql() { return isMicrosoft() || isSybase(); }

  @NotNull
  public static DatabaseFamilyId forDataSource(@NotNull DatabaseSystem o) {
    DatabaseFamilyId result = fromString(o.getDatabaseProductName());
    if (result != UNKNOWN) return result;
    if (o instanceof RawConnectionConfig) {
      result = forConnection((RawConnectionConfig)o);
    }
    return result;
  }

  @NotNull
  public static DatabaseFamilyId forConnection(@NotNull RawConnectionConfig o) {
    DatabaseFamilyId result = fromString(o.getUrl());
    if (result != UNKNOWN) return result;
    return fromString(o.getDriverClass());
  }

  @NotNull
  public static DatabaseFamilyId fromString(@Nullable String text) {
    if (text == null) return UNKNOWN;
    if (StringUtil.containsIgnoreCase(text, "oracle")) return ORACLE;
    if (StringUtil.containsIgnoreCase(text, "MySQL")) return MYSQL;
    if (StringUtil.containsIgnoreCase(text, "postgres")) return POSTGRES;
    if (StringUtil.containsIgnoreCase(text, "microsoft") || StringUtil.containsIgnoreCase(text, "sqlserver")) return MSSQL;
    if (StringUtil.containsIgnoreCase(text, "sybase") || text.startsWith("ASE") || StringUtil.containsIgnoreCase(text, "Adaptive Server")) return SYBASE;
    if (StringUtil.containsIgnoreCase(text, "DB2")) return DB2;
    if (StringUtil.containsIgnoreCase(text, "HSQL")) return HSQLDB;
    if (StringUtil.containsIgnoreCase(text, "H2")) return H2;
    if (StringUtil.containsIgnoreCase(text, "Derby")) return DERBY;
    if (StringUtil.containsIgnoreCase(text, "SQLite")) return SQLITE;
    return UNKNOWN;
  }

}
