/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring;

import com.intellij.ide.TypePresentationService;
import com.intellij.ide.presentation.PresentationProvider;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.pom.PomTargetPsiElement;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.xml.XmlTag;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringBeanPsiTarget;
import com.intellij.spring.model.SpringImplicitBeanMarker;
import com.intellij.spring.model.jam.JamPsiMemberSpringBean;
import com.intellij.spring.model.jam.JamSpringBeanPointer;
import com.intellij.spring.model.scope.SpringBeanScope;
import com.intellij.spring.model.xml.beans.SpringBean;
import com.intellij.spring.model.xml.beans.SpringImport;
import com.intellij.util.xml.DomUtil;
import com.intellij.util.xml.GenericAttributeValue;
import icons.SpringApiIcons;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;

/**
 * @author Yann C&eacute;bron
 */
public class SpringPresentationProvider extends PresentationProvider {

  @Override
  public String getName(Object o) {
    if (o instanceof CommonSpringBean) {
      return getSpringBeanName((CommonSpringBean)o);
    }

    if (o instanceof SpringBeanPointer) {
      return getSpringBeanName((SpringBeanPointer)o);
    }

    if (o instanceof SpringImport) {
      return ((SpringImport)o).getResource().getStringValue();
    }
    return null;
  }

  @Nullable
  @Override
  public Icon getIcon(Object o) {
    return getSpringIcon(o);
  }

  @NotNull
  public static String getSpringBeanName(SpringBeanPointer springBeanPointer) {
    final String pointerName = springBeanPointer.getName();
    return pointerName != null ? pointerName : getSpringBeanName(springBeanPointer.getSpringBean());
  }

  @NotNull
  public static String getSpringBeanName(CommonSpringBean springBean) {
    if (!springBean.isValid()) return "<invalid>";

    final String beanName = springBean.getBeanName();
    if (beanName != null) return beanName;

    if (springBean instanceof SpringBean) {
      final PsiClass beanClass = springBean.getBeanClass();
      if (beanClass != null) return beanClass.getName();
    }

    final String typeName = TypePresentationService.getService().getTypeName(springBean);
    if (typeName != null) return typeName;

    final PsiElement identifyingPsiElement = springBean.getIdentifyingPsiElement();
    if (identifyingPsiElement instanceof PomTargetPsiElement) {
      final PsiElement target = identifyingPsiElement.getNavigationElement();
      if (target instanceof XmlTag) {
        final String name = ((XmlTag)target).getName();
        if (!StringUtil.isEmptyOrSpaces(name)) return "<" + name + " ... />";
      }
    }

    return SpringApiBundle.message("spring.bean.with.unknown.name");
  }

  public static String getSpringBeanLocation(SpringBeanPointer springBeanPointer) {
    if (springBeanPointer instanceof JamSpringBeanPointer) {
      CommonSpringBean commonSpringBean = springBeanPointer.getSpringBean();
      if (commonSpringBean instanceof SpringImplicitBeanMarker) {
        return ((SpringImplicitBeanMarker)commonSpringBean).getProviderName();
      }
    }

    return springBeanPointer.getContainingFile().getName();
  }

  @Nullable
  public static Icon getSpringIcon(@Nullable Object o) {
    if (o instanceof SpringBeanPointer) {
      o = ((SpringBeanPointer)o).getSpringBean();
    }
    else if (o instanceof SpringBeanPsiTarget) {
      o = ((SpringBeanPsiTarget)o).getSpringBean();
    }

    if (!(o instanceof CommonSpringBean)) {
      return null;
    }

    if (o instanceof SpringImplicitBeanMarker) {
      return SpringApiIcons.ImplicitBean;
    }

    if (o instanceof JamPsiMemberSpringBean) {
      return SpringApiIcons.SpringJavaBean;
    }

    if (o instanceof SpringBean && ((SpringBean)o).isValid()) {
      SpringBean springBean = (SpringBean)o;

      if (springBean.isAbstract()) {
        return SpringApiIcons.AbtractBean;
      }

      GenericAttributeValue<SpringBeanScope> scope = springBean.getScope();
      if (DomUtil.hasXml(scope) &&
          scope.getValue() == SpringBeanScope.PROTOTYPE_SCOPE) {
        return SpringApiIcons.PrototypeBean;
      }
    }

    return SpringApiIcons.SpringBean;
  }
}
