/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.contexts.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.NotNullLazyValue;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.util.CachedValue;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.spring.SpringModificationTrackersManager;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.utils.SpringProfileUtils;
import com.intellij.spring.model.xml.context.SpringBeansPackagesScan;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

public class SpringComponentScanModel<T extends SpringBeansPackagesScan> extends ComponentScanPackagesModel {

  private final Set<String> myActiveProfiles;
  private volatile CachedValue<Collection<SpringBeanPointer>> myScannedBeans;

  private final T myComponentScan;

  public SpringComponentScanModel(@NotNull Module module, @NotNull final T componentScan, Set<String> activeProfiles) {
    super(new NotNullLazyValue<Set<PsiPackage>>() {
      @NotNull
      @Override
      protected Set<PsiPackage> compute() {
        return componentScan.getPsiPackages();
      }
    }, module);
    myComponentScan = componentScan;
    myActiveProfiles = activeProfiles;
  }

  public Collection<SpringBeanPointer> getLocalBeans() {
    PsiElement psiElement = myComponentScan.getIdentifyingPsiElement();
    if (psiElement == null) return Collections.emptySet();

    if (myScannedBeans == null) {
      myScannedBeans = CachedValuesManager.getManager(getModule().getProject()).createCachedValue(
        new CachedValueProvider<Collection<SpringBeanPointer>>() {
          @Nullable
          @Override
          public Result<Collection<SpringBeanPointer>> compute() {
            Collection<SpringBeanPointer> pointers = calculateLocalBeans();
            return Result.create(pointers, SpringModificationTrackersManager.getInstance(getModule().getProject()).getOuterModelsDependencies());
          }
        }, false);
    }

    return myScannedBeans.getValue();
  }

  @Override
  protected Collection<SpringBeanPointer> calculateScannedBeans() {
    final Set<CommonSpringBean> elements = myComponentScan.getScannedElements((getModule()));

    List<CommonSpringBean> inActiveProfiles = SpringProfileUtils.filterBeansInActiveProfiles(elements, myActiveProfiles);

    return BeanService.getInstance().mapSpringBeans(inActiveProfiles);
  }

  @NotNull
  @Override
  public Set<String> getActiveProfiles() {
    return myActiveProfiles;
  }

  @NotNull
  @Override
  public Set<PsiFile> getConfigFiles() {
    return Collections.singleton(myComponentScan.getContainingFile());
  }

  @Override
  public boolean hasConfigFile(@NotNull PsiFile configFile) {
    return getConfigFiles().contains(configFile);
  }
}

