/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.contexts.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.progress.ProgressManager;
import com.intellij.psi.PsiFile;
import com.intellij.psi.xml.XmlTag;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.facet.SpringFileSet;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.spring.model.custom.CustomComponentsDiscovererHelper;
import com.intellij.spring.model.xml.context.SpringBeansPackagesScan;
import com.intellij.util.CommonProcessors;
import com.intellij.util.Processor;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.hash.HashSet;
import com.intellij.xml.util.PsiElementPointer;
import gnu.trove.THashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public abstract class SpringModel implements CommonSpringModel {

  @Nullable
  private final Module myModule;

  private Set<SpringModel> myDependencies = Collections.emptySet();

  @Nullable
  private final SpringFileSet myFileSet;

  protected SpringModel(@Nullable Module module) {
    this(module, null);
  }

  protected SpringModel(@Nullable final Module module,
                        @Nullable final SpringFileSet fileSet) {
    myFileSet = fileSet;
    myModule = module;
  }

  public SpringFileSet getFileSet() {
    return myFileSet;
  }

  @NotNull
  public Set<SpringModel> getDependencies() {
    return myDependencies;
  }

  public void setDependencies(@NotNull final Set<SpringModel> dependencies) {
    myDependencies = dependencies;
  }

  @NotNull
  public final Set<CommonSpringModel> getModelsToProcess() {
    Set<CommonSpringModel> models = new LinkedHashSet<CommonSpringModel>();
    ContainerUtil.addAllNotNull(models, getModelsToProcess(true));

    if (myModule != null && !myModule.isDisposed()) {
      ContainerUtil.addAllNotNull(models, CustomComponentsDiscovererHelper.getInstance(myModule).getCustomModels());
    }
    return models;
  }

  @NotNull
  public abstract Set<CommonSpringModel> getModelsToProcess(boolean checkActiveProfiles);

  @NotNull
  public Collection<SpringBeanPointer> getAllDomBeans() {
    final List<SpringBeanPointer> pointers = new ArrayList<SpringBeanPointer>();

    processModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel model) {
        pointers.addAll(model.getAllDomBeans());
        return true;
      }
    });

    return pointers;
  }

  @NotNull
  public Set<String> getAllBeanNames(@NotNull final String beanName) {
    final Set<String> allOtherNames = new THashSet<String>();

    processModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel model) {
        final Set<String> otherNames = model.getAllBeanNames(beanName);
        if (otherNames.size() > 1) {
          allOtherNames.addAll(otherNames);
        }
        return true;
      }
    });
    return allOtherNames.size() > 0 ? allOtherNames : Collections.singleton(beanName);
  }

  public Collection<PsiElementPointer> getDuplicatedNames(@NotNull final String beanName) {
    final List<PsiElementPointer> pointers = new SmartList<PsiElementPointer>();

    processModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel model) {
        pointers.addAll(model.getDuplicatedNames(beanName));   // todo incorrect! add bean names from additional models
        return true;
      }
    });
    return pointers;
  }

  @NotNull
  public Collection<SpringBeanPointer> getAllCommonBeans() {
    final Collection<SpringBeanPointer> pointers = new ArrayList<SpringBeanPointer>();

    processModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel model) {
        pointers.addAll(model.getAllCommonBeans());
        return true;
      }
    });

    return pointers;
  }

  @NotNull
  public List<SpringBeanPointer> findQualifiedBeans(@NotNull final SpringQualifier qualifier) {
    final List<SpringBeanPointer> pointers = new ArrayList<SpringBeanPointer>();

    processModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel model) {
        pointers.addAll(model.findQualifiedBeans(qualifier));
        return true;
      }
    });

    return pointers;
  }

  @Override
  public boolean processByClass(@NotNull final SpringModelSearchParameters.BeanClass params,
                                @NotNull Processor<SpringBeanPointer> processor) {
    if (!params.canSearch()) return true;
    final VisitedModelsDelegateProcessor delegateProcessor = new VisitedModelsDelegateProcessor(processor);
    return processModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel commonModel) {
        if (delegateProcessor.hasBeenVisited(commonModel)) return true;

        delegateProcessor.addVisited(commonModel);
        if (!commonModel.processByClass(params, delegateProcessor)) return false;

        return true;
      }
    });
  }

  @Override
  public boolean processByName(@NotNull final SpringModelSearchParameters.BeanName params,
                               @NotNull Processor<SpringBeanPointer> processor) {
    if (!params.canSearch()) return true;

    final VisitedModelsDelegateProcessor delegateProcessor = new VisitedModelsDelegateProcessor(processor);
    return processModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel commonModel) {
        if (delegateProcessor.hasBeenVisited(commonModel)) return true;

        delegateProcessor.addVisited(commonModel);
        if (!commonModel.processByName(params, delegateProcessor)) return false;

        return true;
      }
    });
  }

  @Nullable
  public Module getModule() {
    return myModule;
  }

  public String toString() {
    final SpringFileSet fileSet = getFileSet();
    return getClass().getName() + (fileSet != null ? " fileset=" + fileSet.getId() : "");
  }

  @NotNull
  @Override
  public List<SpringBeanPointer> getPlaceholderConfigurers() {
    final List<SpringBeanPointer> pointers = new SmartList<SpringBeanPointer>();
    processModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel model) {
        pointers.addAll(model.getPlaceholderConfigurers());
        return true;
      }
    });
    return pointers;
  }

  @NotNull
  @Override
  public List<SpringBeansPackagesScan> getComponentScans() {
    final List<SpringBeansPackagesScan> pointers = new SmartList<SpringBeansPackagesScan>();

    processModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel model) {
        pointers.addAll(model.getComponentScans());
        return true;
      }
    });


    return pointers;
  }

  protected boolean processModels(@NotNull Processor<CommonSpringModel> processor) {
    return processModels(this, new HashSet<CommonSpringModel>(), processor);
  }

  private static boolean processModels(@NotNull CommonSpringModel model,
                                       @NotNull Set<CommonSpringModel> visited,
                                       @NotNull Processor<CommonSpringModel> processor) {
    if (!visited.contains(model)) {
      visited.add(model);
      ProgressManager.checkCanceled();
      if (model instanceof SpringModel) {
        for (CommonSpringModel commonModel : ((SpringModel)model).getModelsToProcess()) {
          if (!processModels(commonModel, visited, processor)) return false;
        }
      }
      else {
        return processor.process(model);
      }
    }
    return true;
  }

  @NotNull
  @Override
  public List<SpringBeanPointer> getAnnotationConfigApplicationContexts() {
    final List<SpringBeanPointer> pointers = new SmartList<SpringBeanPointer>();
    processModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel model) {
        pointers.addAll(model.getAnnotationConfigApplicationContexts());
        return true;
      }
    });
    return pointers;
  }

  @Nullable
  @Override
  public Set<String> getActiveProfiles() {
    if (myFileSet == null) {
      return null;
    }
    return myFileSet.getActiveProfiles();
  }

  public Collection<XmlTag> getCustomBeanCandidates(final String id) {
    final Collection<XmlTag> tags = new SmartList<XmlTag>();

    processModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel model) {
        tags.addAll(model.getCustomBeanCandidates(id));
        return true;
      }
    });

    return tags;
  }

  @NotNull
  @Override
  public List<SpringBeanPointer> getDescendants(@NotNull final SpringBeanPointer context) {
    final List<SpringBeanPointer> pointers = new SmartList<SpringBeanPointer>();
    processModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel model) {
        pointers.addAll(model.getDescendants(context));
        return true;
      }
    });

    return pointers;
  }

  @NotNull
  @Override
  public Set<PsiFile> getConfigFiles() {
    final Set<PsiFile> files = new LinkedHashSet<PsiFile>();

    final Processor<CommonSpringModel> collectConfigFiles = new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel commonSpringModel) {
        files.addAll(commonSpringModel.getConfigFiles());
        return true;
      }
    };

    processConfigFiles(collectConfigFiles);

    return files;
  }

  @Override
  public boolean hasConfigFile(@NotNull final PsiFile configFile) {
    CommonProcessors.FindProcessor<CommonSpringModel> findProcessor = new CommonProcessors.FindProcessor<CommonSpringModel>() {
      @Override
      protected boolean accept(CommonSpringModel model) {
        return model.hasConfigFile(configFile);
      }
    };
    processConfigFiles(findProcessor);
    return findProcessor.isFound();
  }

  protected boolean processConfigFiles(final Processor<CommonSpringModel> collectConfigFiles) {
    return processModels(new Processor<CommonSpringModel>() {
      @Override
      public boolean process(CommonSpringModel commonModel) {
        if (!collectConfigFiles.process(commonModel)) return false;

        if (commonModel instanceof CachedLocalModel) {
          if (!((CachedLocalModel)commonModel).processRelatedModels(collectConfigFiles)) return false;
        }

        return true;
      }
    });
  }

  public static final CommonSpringModel UNKNOWN = new CommonSpringModel() {

    @Override
    public boolean processByClass(@NotNull SpringModelSearchParameters.BeanClass params,
                                  @NotNull Processor<SpringBeanPointer> processor) {
      return true;
    }

    @Override
    public boolean processByName(@NotNull SpringModelSearchParameters.BeanName params,
                                 @NotNull Processor<SpringBeanPointer> processor) {
      return true;
    }

    @NotNull
    @Override
    public Collection<SpringBeanPointer> getAllDomBeans() {
      return Collections.emptySet();
    }

    @NotNull
    @Override
    public Collection<SpringBeanPointer> getAllCommonBeans() {
      return Collections.emptySet();
    }

    @NotNull
    @Override
    public Set<String> getAllBeanNames(@NotNull String beanName) {
      return Collections.emptySet();
    }

    @Override
    public Collection<PsiElementPointer> getDuplicatedNames(@NotNull String beanName) {
      return Collections.emptySet();
    }

    @NotNull
    @Override
    public List<SpringBeanPointer> getPlaceholderConfigurers() {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public List<SpringBeansPackagesScan> getComponentScans() {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public List<SpringBeanPointer> getAnnotationConfigApplicationContexts() {
      return Collections.emptyList();
    }

    @Override
    public Collection<XmlTag> getCustomBeanCandidates(String id) {
      return Collections.emptyList();
    }

    @Override
    public Module getModule() {
      return null;
    }


    @Override
    public Set<String> getActiveProfiles() {
      return Collections.emptySet();
    }

    @NotNull
    @Override
    public Set<String> getAllProfiles() {
      return Collections.emptySet();
    }

    @NotNull
    @Override
    public Set<PsiFile> getConfigFiles() {
      return Collections.emptySet();
    }

    @Override
    public boolean hasConfigFile(@NotNull PsiFile configFile) {
      return false;
    }

    @NotNull
    @Override
    public List<SpringBeanPointer> findQualifiedBeans(@NotNull SpringQualifier qualifier) {
      return Collections.emptyList();
    }

    @NotNull
    @Override
    public List<SpringBeanPointer> getDescendants(@NotNull SpringBeanPointer context) {
      return Collections.emptyList();
    }
  };
}
