/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model;

import com.intellij.ide.presentation.Presentation;
import com.intellij.openapi.util.Comparing;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiFile;
import com.intellij.spring.SpringPresentationProvider;
import com.intellij.util.Function;
import com.intellij.xml.util.PsiElementPointer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Comparator;

/**
 * Lightweight reference to Spring bean.
 *
 * @author peter
 * @see BeanService#createSpringBeanPointer(CommonSpringBean)
 */
@Presentation(provider = SpringPresentationProvider.class)
public interface SpringBeanPointer<T extends CommonSpringBean> extends PsiElementPointer {

  Function<SpringBeanPointer, CommonSpringBean> TO_BEAN = new Function<SpringBeanPointer, CommonSpringBean>() {
    @Override
    public CommonSpringBean fun(SpringBeanPointer pointer) {
      return pointer.getSpringBean();
    }
  };

  /**
   * Sort beans for UI purposes.
   *
   * @since 14
   */
  Comparator<SpringBeanPointer> DISPLAY_COMPARATOR = new Comparator<SpringBeanPointer>() {
    @Override
    public int compare(SpringBeanPointer o1, SpringBeanPointer o2) {
      return Comparing.compare(SpringPresentationProvider.getSpringBeanName(o1),
                               SpringPresentationProvider.getSpringBeanName(o2));
    }
  };

  /**
   * Defined bean name.
   * <p/>
   * For presentation purposes use {@link com.intellij.spring.SpringPresentationProvider#getSpringBeanName(com.intellij.spring.model.SpringBeanPointer)}.
   *
   * @return {@code null} if no name could be determined.
   */
  @Nullable
  String getName();

  String[] getAliases();

  @NotNull
  T getSpringBean();

  boolean isValid();

  boolean isReferenceTo(@Nullable CommonSpringBean springBean);

  SpringBeanPointer derive(@NotNull String name);

  /**
   * @return {@code this} or underlying pointer (e.g. for derived pointers).
   */
  @NotNull
  SpringBeanPointer getBasePointer();

  @Nullable
  PsiClass getBeanClass();

  PsiClass[] getEffectiveBeanType();

  PsiFile getContainingFile();

  /**
   * Only for {@link com.intellij.spring.model.xml.beans.SpringBean}.
   *
   * @return {@code true} if bean is abstract.
   */
  boolean isAbstract();

  /**
   * Only for {@link com.intellij.spring.model.xml.beans.SpringBean}.
   *
   * @return Pointer to parent bean if defined.
   */
  @Nullable
  SpringBeanPointer getParentPointer();
}