/*
 * Copyright 2000-2013 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.converters;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.extensions.Extensions;
import com.intellij.openapi.util.Condition;
import com.intellij.util.containers.MultiMap;
import com.intellij.util.xml.Converter;
import com.intellij.util.xml.GenericDomValue;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;

public class CustomConverterRegistry {

  public interface Provider extends Condition<GenericDomValue> {

    ExtensionPointName<Provider> EXTENSION_POINT_NAME = ExtensionPointName.create("com.intellij.spring.customConverterProvider");

    Converter getConverter();

    Class getConverterClass();
  }

  public static CustomConverterRegistry getRegistry() {
    return ourInstance;
  }

  private final static CustomConverterRegistry ourInstance = new CustomConverterRegistry();

  private final MultiMap<Class, Provider> myCustomConverters;

  private CustomConverterRegistry() {
    myCustomConverters = new MultiMap<Class, Provider>();
    Provider[] providers = Extensions.getExtensions(Provider.EXTENSION_POINT_NAME);
    for (Provider provider : providers) {
      myCustomConverters.putValue(provider.getConverterClass(), provider);
    }
  }

  @Nullable
  public Converter getCustomConverter(Class aClass, GenericDomValue context) {
    Collection<Provider> providers = myCustomConverters.get(aClass);
    for (Provider provider : providers) {
      if (provider.value(context)) {
        return provider.getConverter();
      }
    }

    return null;
  }

}
