/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.spring.model.converters;

import com.intellij.openapi.util.TextRange;
import com.intellij.openapi.util.io.FileUtil;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiElementResolveResult;
import com.intellij.psi.PsiFileSystemItem;
import com.intellij.psi.ResolveResult;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.FileReference;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.FileReferenceSet;
import com.intellij.psi.search.PsiFileSystemItemProcessor;
import com.intellij.spring.SpringApiBundle;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.regex.Pattern;

/**
 * @author Dmitry Avdeev
 */
public class PatternFileReferenceSet extends FileReferenceSet {

  public PatternFileReferenceSet(final String str, @NotNull PsiElement element, final int offset) {
    this(str, element, offset, true);
  }

  public PatternFileReferenceSet(@NotNull String str, @NotNull PsiElement element, final int offset, boolean endingSlashNotAllowed) {
    super(str, element, offset, null, true, endingSlashNotAllowed);
  }

  @Override
  public FileReference createFileReference(final TextRange range, final int index, final String text) {
    if (!isAntPattern(text)) return super.createFileReference(range, index, text);

    return new PatternFileReference(this, range, index, text);
  }

  // @see org.springframework.util.AntPathMatcher#isPattern
  public static boolean isAntPattern(final String str) {
    return (str.indexOf('*') != -1 || str.indexOf('?') != -1);
  }

  @Override
  protected boolean isSoft() {
    return true;
  }

  /**
   * @author Dmitry Avdeev
   */
  public static class PatternFileReference extends FileReference {
    public PatternFileReference(final FileReferenceSet referenceSet, final TextRange range, final int index, final String text) {
      super(referenceSet, range, index, text);
    }

    @Override
    protected void innerResolveInContext(@NotNull final String text,
                                         @NotNull final PsiFileSystemItem context,
                                         final Collection<ResolveResult> result, final boolean caseSensitive) {


      if (text.equals("**")) {
        addDirectoryResolves(context, result);
      }
      else {
        final String patternText = FileUtil.convertAntToRegexp(text);
        final Pattern pattern = Pattern.compile(patternText);

        final PsiElement[] psiElements = context.getChildren();
        for (PsiElement psiElement : psiElements) {
          if (psiElement instanceof PsiFileSystemItem) {
            if (pattern.matcher(((PsiFileSystemItem)psiElement).getName()).matches()) {
              result.add(new PsiElementResolveResult(psiElement));
            }
          }
        }
      }
    }

    private static void addDirectoryResolves(final PsiElement context, final Collection<ResolveResult> result) {
      if (!(context instanceof PsiFileSystemItem)) {
        return;
      }

      final PsiFileSystemItem fileSystemItem = (PsiFileSystemItem)context;
      if (fileSystemItem.isDirectory()) {
        result.add(new PsiElementResolveResult(context));

        fileSystemItem.processChildren(new PsiFileSystemItemProcessor() {

          @Override
          public boolean acceptItem(String name, boolean isDirectory) {
            return isDirectory;
          }

          @Override
          public boolean execute(@NotNull PsiFileSystemItem element) {
            final VirtualFile virtualFile = element.getVirtualFile();
            if (virtualFile != null) {
              addDirectoryResolves(element.getManager().findDirectory(virtualFile), result);
            }
            return true;
          }
        });
      }
    }

    @Override
    @NotNull
    public String getUnresolvedMessagePattern() {
      return SpringApiBundle.message("pattern.fileset.no.matching.files");
    }
  }
}
