/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.converters;

import com.intellij.codeInsight.daemon.EmptyResolveMessageProvider;
import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.codeInsight.lookup.LookupElementBuilder;
import com.intellij.codeInspection.LocalQuickFix;
import com.intellij.codeInspection.LocalQuickFixProvider;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.util.PsiFormatUtil;
import com.intellij.psi.util.PsiFormatUtilBase;
import com.intellij.spring.SpringApiBundle;
import com.intellij.util.ArrayUtil;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.xml.ConvertContext;
import com.intellij.util.xml.Converter;
import com.intellij.util.xml.CustomReferenceConverter;
import com.intellij.util.xml.GenericDomValue;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * @author Dmitry Avdeev
 */
public abstract class PsiMethodConverter extends Converter<PsiMethod> implements CustomReferenceConverter<PsiMethod> {

  protected final static Object[] EMPTY_ARRAY = ArrayUtil.EMPTY_OBJECT_ARRAY;

  private final MethodAccepter myMethodAccepter;

  protected PsiMethodConverter(MethodAccepter accepter) {
    myMethodAccepter = accepter;
  }

  protected PsiMethodConverter() {
    this(new MethodAccepter());
  }

  protected static class MethodAccepter {
    public boolean accept(PsiMethod method) {
      return !method.isConstructor() &&
             method.hasModifierProperty(PsiModifier.PUBLIC) &&
             !method.hasModifierProperty(PsiModifier.STATIC);
    }
  }

  public PsiMethod fromString(@Nullable final String methodName, final ConvertContext context) {
    if (StringUtil.isEmpty(methodName)) {
      return null;
    }
    final PsiClass psiClass = getPsiClass(context);
    if (psiClass == null) {
      return null;
    }
    final PsiMethod[] psiMethods = getMethodCandidates(methodName, psiClass);
    if (psiMethods.length == 0) {
      return null;
    }
    final MethodAccepter accepter = getMethodAccepter(context, false);
    for (PsiMethod method : psiMethods) {
      if (accepter.accept(method)) {
        return method;
      }
    }
    return null;
  }

  protected abstract PsiMethod[] getMethodCandidates(String methodIdentificator, PsiClass psiClass);

  public String toString(@Nullable final PsiMethod psiMethods, final ConvertContext context) {
    return null;
  }

  @Nullable
  protected abstract PsiClass getPsiClass(final ConvertContext context);

  protected MethodAccepter getMethodAccepter(ConvertContext context, final boolean forCompletion) {
    return myMethodAccepter;
  }

  protected Object[] getVariants(ConvertContext context) {
    final PsiClass psiClass = getPsiClass(context);
    if (psiClass == null) {
      return EMPTY_ARRAY;
    }
    final MethodAccepter methodAccepter = getMethodAccepter(context, true);
    final List<LookupElement> result = new ArrayList<LookupElement>();
    Collection<HierarchicalMethodSignature> allMethodSigs = psiClass.getVisibleSignatures();

    for (HierarchicalMethodSignature signature : allMethodSigs) {
      final PsiMethod method = signature.getMethod();
      if (methodAccepter.accept(method)) {
        String tail = PsiFormatUtil.formatMethod(method,
                                                 PsiSubstitutor.EMPTY,
                                                 PsiFormatUtilBase.SHOW_PARAMETERS,
                                                 PsiFormatUtilBase.SHOW_NAME | PsiFormatUtilBase.SHOW_TYPE);
        LookupElementBuilder builder = LookupElementBuilder.create(method, getMethodIdentificator(method))
          .withIcon(method.getIcon(0))
          .withStrikeoutness(method.isDeprecated())
          .withTailText(tail);
        final PsiType returnType = method.getReturnType();
        if (returnType != null) {
          builder = builder.withTypeText(returnType.getPresentableText());
        }
        result.add(builder);
      }
    }
    return ArrayUtil.toObjectArray(result);
  }

  protected abstract String getMethodIdentificator(PsiMethod method);

  @NotNull
  public PsiReference[] createReferences(final GenericDomValue<PsiMethod> genericDomValue,
                                         final PsiElement element,
                                         final ConvertContext context) {

    return new PsiReference[]{new MyReference(element, genericDomValue, context)};
  }

  protected class MyReference extends PsiReferenceBase<PsiElement> implements EmptyResolveMessageProvider, LocalQuickFixProvider {
    private final GenericDomValue<PsiMethod> myGenericDomValue;
    private final ConvertContext myContext;

    public MyReference(final PsiElement element,
                       final GenericDomValue<PsiMethod> genericDomValue,
                       ConvertContext context) {
      super(element);
      myGenericDomValue = genericDomValue;
      myContext = context;
    }

    @NotNull
    public Object[] getVariants() {
      return PsiMethodConverter.this.getVariants(myContext);
    }

    @Nullable
    public PsiElement resolve() {
      return myGenericDomValue.getValue();
    }

    public boolean isSoft() {
      return true;
    }

    public PsiElement bindToElement(@NotNull final PsiElement element) throws IncorrectOperationException {
      if (!(element instanceof PsiMethod)) {
        throw new IncorrectOperationException("PsiMethod expected, but found: " + element);
      }
      final PsiMethod psiMethod = (PsiMethod)element;
      myGenericDomValue.setStringValue(getMethodIdentificator(psiMethod));
      return psiMethod;
    }

    public LocalQuickFix[] getQuickFixes() {
      return PsiMethodConverter.this.getQuickFixes(myContext);
    }

    @NotNull
    public String getUnresolvedMessagePattern() {
      return SpringApiBundle.message("cannot.resolve.method", myGenericDomValue.getStringValue());
    }
  }

  public LocalQuickFix[] getQuickFixes(final ConvertContext context) {
    return LocalQuickFix.EMPTY_ARRAY;
  }
}
