/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.testContexts;

import com.intellij.jam.JamElement;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamAttributeMeta;
import com.intellij.jam.reflect.JamClassMeta;
import com.intellij.jam.reflect.JamStringAttributeMeta;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiClass;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.util.containers.HashSet;
import org.jetbrains.annotations.NotNull;

import java.util.Set;

/**
 * @since 16
 */
public class SpringActiveProfiles extends CommonModelElement.PsiBase implements JamElement {

  private static final JamStringAttributeMeta.Collection<String> VALUE_ATTR_META = JamAttributeMeta.collectionString("value");
  private static final JamStringAttributeMeta.Collection<String> PROFILES_ATTR_META = JamAttributeMeta.collectionString("profiles");

  public static final JamAnnotationMeta ANNOTATION_META =
    new JamAnnotationMeta(SpringAnnotationsConstants.ACTIVE_PROFILES).addAttribute(PROFILES_ATTR_META)
      .addAttribute(VALUE_ATTR_META);

  public static final JamClassMeta<SpringActiveProfiles> META =
    new JamClassMeta<SpringActiveProfiles>(SpringActiveProfiles.class).addAnnotation(ANNOTATION_META);

  private final PsiClass myPsiElement;

  public SpringActiveProfiles(@NotNull PsiClass psiElement) {
    myPsiElement = psiElement;
  }

  @NotNull
  public Set<String> getActiveProfiles() {
    Set<String> profiles = new HashSet<String>();

    addProfiles(profiles, VALUE_ATTR_META);
    addProfiles(profiles, PROFILES_ATTR_META);

    return profiles;
  }

  private void addProfiles(@NotNull Set<String> profiles,
                           @NotNull JamStringAttributeMeta.Collection<String> attrMeta) {
    for (JamStringAttributeElement<String> element : ANNOTATION_META.getAttribute(getPsiElement(), attrMeta)) {
      if (element != null) {
        final String value = element.getValue();
        if (StringUtil.isNotEmpty(value)) profiles.add(value);
      }
    }
  }

  @NotNull
  @Override
  public PsiClass getPsiElement() {
    return myPsiElement;
  }
}
