/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.utils;

import com.intellij.ide.projectView.impl.ProjectRootsUtil;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.util.text.DelimitedListProcessor;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.PackageReferenceSet;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.PatternPackageReferenceSet;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.util.PairProcessor;
import com.intellij.util.SmartList;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

public class SpringReferenceUtils {
  @NotNull
  public static PsiReference[] getPsiPackagesReferences(@NotNull final PsiElement element,
                                                        @Nullable final String text,
                                                        final int startInElement) {
    return getPsiPackagesReferences(element, text, startInElement, ",; \n\t");
  }

  public static PsiReference[] getPsiPackagesReferences(@NotNull final PsiElement element,
                                                        @Nullable final String text,
                                                        final int startInElement,
                                                        String delimiters) {
    return getPsiPackagesReferences(element, text, startInElement, delimiters, getResolveScope(element));
  }

  public static PsiReference[] getPsiPackagesReferences(@NotNull final PsiElement element,
                                                        @Nullable final String text,
                                                        final int startInElement,
                                                        String delimiters, @NotNull final GlobalSearchScope scope) {
    if (text == null) {
      return PsiReference.EMPTY_ARRAY;
    }

    final List<PsiReference> list = new SmartList<PsiReference>();
    new DelimitedListProcessor(delimiters) {
      protected void processToken(final int start, final int end, final boolean delimitersOnly) {
        final String packageName = text.substring(start, end);
        final String trimmedPackageName = packageName.trim();
        int spacesNumber = packageName.indexOf(trimmedPackageName);
        final PackageReferenceSet referenceSet =
          new PatternPackageReferenceSet(trimmedPackageName, element, start + spacesNumber + startInElement, scope);
        list.addAll(referenceSet.getReferences());
      }
    }.processText(text);
    return list.isEmpty() ? PsiReference.EMPTY_ARRAY : list.toArray(new PsiReference[list.size()]);
  }

  private static GlobalSearchScope getResolveScope(@NotNull PsiElement element) {
    Module module = ModuleUtilCore.findModuleForPsiElement(element);
    if (module == null) {
      return GlobalSearchScope.allScope(element.getProject());
    }

    final boolean inTestSource = ProjectRootsUtil.isInTestSource(element.getContainingFile());
    return module.getModuleRuntimeScope(inTestSource);
  }

  public static boolean processSeparatedString(final String str, final String delimiter, final PairProcessor<String, Integer> processor) {
    if (str == null) return true;
    if (StringUtil.isEmptyOrSpaces(str)) return processor.process(str, 0);

    int pos = 0;
    while (true) {
      int index = str.indexOf(delimiter, pos);
      if (index == -1) break;
      final int nextPos = index + delimiter.length();
      String token = str.substring(pos, index);
      if (token.length() != 0) {
        if (!processor.process(token.trim(), pos + token.indexOf(token.trim()))) return false;
      }
      pos = nextPos;
    }
    if (pos < str.length()) {
      final String s = str.substring(pos, str.length());
      if (!processor.process(s.trim(), pos + s.indexOf(s.trim()))) return false;
    }
    return true;
  }
}
