/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.spi;

import com.intellij.codeInsight.completion.JavaLookupElementBuilder;
import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.jam.JamConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.roots.ModuleFileIndex;
import com.intellij.openapi.roots.ModuleRootManager;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.util.Function;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;

/**
 * Provides class reference to valid values configured via given config key in {@code spring.factories}.
 *
 * @see SpringSpiManager#getClassesListValue(boolean, String)
 * @since 15
 */
public class SpringSpiClassesListJamConverter extends JamConverter<PsiClass> {

  private final String myConfigKey;

  public SpringSpiClassesListJamConverter(String configKey) {
    myConfigKey = configKey;
  }

  @Nullable
  @Override
  public PsiClass fromString(@Nullable final String s, JamStringAttributeElement<PsiClass> context) {
    if (StringUtil.isEmptyOrSpaces(s)) return null;

    final PsiReference[] references = createReferences(context);
    if (references.length != 1) return null;
    return (PsiClass)references[0].resolve();
  }

  @NotNull
  @Override
  public PsiReference[] createReferences(final JamStringAttributeElement<PsiClass> context) {
    final PsiLiteral literal = context.getPsiLiteral();
    if (literal == null) return PsiReference.EMPTY_ARRAY;

    return new PsiReference[]{new SpringSpiClassReference(myConfigKey, literal, context.getStringValue())};
  }


  public static class SpringSpiClassReference extends PsiReferenceBase<PsiElement> {

    private final String myConfigKey;
    private final String myText;

    private SpringSpiClassReference(String configKey, @NotNull PsiLiteral literal, String text) {
      super(literal);
      myConfigKey = configKey;
      myText = text;
    }

    @Nullable
    @Override
    public PsiElement resolve() {
      if (StringUtil.isEmptyOrSpaces(myText)) return null;

      return ContainerUtil.find(getRelevantClasses(getElement()), new Condition<PsiClass>() {
        @Override
        public boolean value(PsiClass psiClass) {
          return myText.equals(psiClass.getQualifiedName());
        }
      });
    }

    @NotNull
    @Override
    public Object[] getVariants() {
      final List<PsiClass> classes = getRelevantClasses(getElement());
      return ContainerUtil.map2Array(classes, LookupElement.class, new Function<PsiClass, LookupElement>() {
        @Override
        public LookupElement fun(PsiClass psiClass) {
          return JavaLookupElementBuilder.forClass(psiClass, psiClass.getQualifiedName(), true)
            .withPresentableText(psiClass.getName());
        }
      });
    }

    private List<PsiClass> getRelevantClasses(PsiElement literal) {
      if (literal == null) return Collections.emptyList();

      final Module module = ModuleUtilCore.findModuleForPsiElement(literal);
      if (module == null) return Collections.emptyList();

      final ModuleFileIndex moduleFileIndex = ModuleRootManager.getInstance(module).getFileIndex();
      final boolean isInTest = moduleFileIndex.isInTestSourceContent(literal.getContainingFile().getOriginalFile().getVirtualFile());
      return SpringSpiManager.getInstance(module).getClassesListValue(isInTest, myConfigKey);
    }
  }
}
