/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.psi.css;

import com.intellij.css.util.CssConstants;
import com.intellij.css.util.CssPsiUtil;
import com.intellij.lang.Language;
import com.intellij.lang.css.CSSLanguage;
import com.intellij.lang.css.CssLanguageProperties;
import com.intellij.openapi.components.ServiceManager;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.util.ArrayUtil;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class CssElementFactory {
  private final PsiManager myManager;

  public CssElementFactory(PsiManager _manager) {
    myManager = _manager;
  }

  public static CssElementFactory getInstance(@NotNull Project project) {
    return ServiceManager.getService(project, CssElementFactory.class);
  }

  @NotNull
  public CssString createString(char quoteChar, String value, @Nullable Language language) {
    String stringText = quoteChar + value + quoteChar;
    language = getLanguage(language);
    String declarationsTerminator = language instanceof CssLanguageProperties
                                    ? ((CssLanguageProperties)language).getDeclarationsTerminator()
                                    : CssConstants.SEMICOLON;
    String importText = "@import " + stringText + declarationsTerminator;
    CssStylesheet stylesheet = createStylesheet(importText, language);
    final CssImport imports = PsiTreeUtil.findChildOfType(stylesheet, CssImport.class);
    final CssString cssString = PsiTreeUtil.getChildOfType(imports, CssString.class);
    if (cssString == null) {
      throw new IllegalArgumentException("Can't create css string with text: " + stringText);
    }
    return cssString;
  }

  @NotNull
  public PsiElement createStringToken(char quoteChar, String value, @Nullable Language language) {
    final CssString string = createString(quoteChar, value, language);
    assert string.getChildren().length > 0;
    return string.getFirstChild();
  }

  @NotNull
  public CssStylesheet createStylesheet(@NotNull String text, @Nullable Language language) {
    language = getLanguage(language);
    final PsiFile fileFromText = PsiFileFactory.getInstance(myManager.getProject()).createFileFromText("_dummy_", language, text, false, true);
    if (!(fileFromText instanceof CssFile)) {
      throw new IllegalArgumentException("Can't create ruleset for given file type: " + fileFromText.getFileType().getName());
    }
    CssStylesheet stylesheet = ((CssFile)fileFromText).getStylesheet();
    if (stylesheet == null) {
      throw new IllegalArgumentException("Can't create css stylesheet from text: " + text);
    }
    return stylesheet;
  }

  @NotNull
  public CssRuleset createRuleset(@NotNull @NonNls String text, @Nullable Language language) {
    final CssRuleset[] rulesets = createStylesheet(text, language).getRulesets();
    if (rulesets.length == 0) {
      throw new IllegalArgumentException("Can't create ruleset from text: " + text);
    }
    final CssRuleset ruleset = rulesets[0];
    if (ruleset == null) {
      throw new IllegalArgumentException("Can't create ruleset from text: " + text);
    }
    return ruleset;
  }

  @NotNull
  public CssRuleset createRuleset(@NotNull String selector, @NotNull CssDeclaration[] declarations, @Nullable Language language) {
    language = getLanguage(language);
    String declarationsTerminator = CssPsiUtil.getDeclarationsTerminator(language);
    StringBuilder blockText = new StringBuilder();
    for (CssDeclaration declaration : declarations) {
      blockText.append("  ").append(declaration.getPropertyName()).append(": ");
      final CssTermList declarationValue = declaration.getValue();
      if (declarationValue != null) {
        blockText.append(declarationValue.getText());
      }
      blockText.append(declarationsTerminator).append("\n");
    }
    return createRuleset(createRulesetText(selector, blockText, language).toString(), language);
  }

  @NotNull
  public CssDeclaration[] createProperties(@NotNull String text, @Nullable Language language) {
    final StringBuilder rulesetText = createRulesetText("div", new StringBuilder(text), language);
    CssBlock block = createRuleset(rulesetText.toString(), language).getBlock();
    if (block == null) {
      throw new IllegalArgumentException("Can't create css declarations from text: " + text);
    }
    return block.getDeclarations();
  }

  @NotNull
  public CssDeclaration createProperty(@NotNull String propertyText, @Nullable Language language) {
    CssDeclaration[] properties = createProperties(propertyText, language);
    if (properties.length == 0) {
      throw new IllegalArgumentException("Can't create css property from text: " + propertyText);
    }
    return properties[0];
  }

  @NotNull
  public CssDeclaration createProperty(@NotNull String name, @NotNull String value, @Nullable Language language) {
    return createProperty(name + ": " + value + CssPsiUtil.getDeclarationsTerminator(getLanguage(language)), language);
  }

  @Nullable
  public CssMedia createMedia(@NotNull String mediaQuery, @Nullable Language language) {
    String mediaText = createRulesetText("@media " + mediaQuery + "{}", new StringBuilder(), language).toString();
    CssStylesheet stylesheet = createStylesheet(mediaText, language);
    for (CssAtRule rule : stylesheet.getRulesetList().getAtRules()) {
      if (rule instanceof CssMedia) {
        return (CssMedia)rule;
      }
    }
    return null;
  }

  @NotNull
  public CssMediaFeature createMediaFeature(@NotNull String name, @NotNull String value, @Nullable Language language) {
    String mediaText = createRulesetText("@media screen and (" + name + ":" + value + ")", new StringBuilder(), language).toString();
    CssStylesheet stylesheet = createStylesheet(mediaText, language);
    for (CssAtRule rule : stylesheet.getRulesetList().getAtRules()) {
      if (rule instanceof CssMedia) {
        CssMediumList mediumList = ((CssMedia)rule).getMediumList();
        if (mediumList != null) {
          CssMediaQuery mediaQuery = ArrayUtil.getFirstElement(mediumList.getMediaQueries());
          if (mediaQuery != null) {
            CssMediaExpression mediaExpression = ArrayUtil.getFirstElement(mediaQuery.getExpressions());
            CssMediaFeature mediaFeature = PsiTreeUtil.getChildOfType(mediaExpression, CssMediaFeature.class);
            if (mediaFeature != null) {
              return mediaFeature;
            }
          }
        }
      }
    }

    throw new IllegalArgumentException("Can't create css media feature: " + mediaText);
  }

  @NotNull
  public CssTermList createPropertyValue(@NotNull String value, @Nullable Language language) {
    final CssTermList propertyValue = createProperty("width", value, language).getValue();
    if (propertyValue == null) {
      throw new IllegalArgumentException("Can't create css property value: " + value);
    }
    return propertyValue;
  }

  @NotNull
  public CssDescriptorOwner createFunction(@NotNull String name, @Nullable Language language) {
    CssTerm term = createTerm(name + "(param)", language);
    CssFunction function = PsiTreeUtil.findChildOfType(term, CssFunction.class);
    if (function == null) {
      throw new IllegalArgumentException("Can't create css function with name: " + name);
    }
    return function;
  }

  @NotNull
  public CssTerm createTerm(@NotNull String value, @Nullable Language language) {
    CssTermList propertyValue = createPropertyValue(value, language);
    final CssTerm term = PsiTreeUtil.findChildOfType(propertyValue, CssTerm.class);
    if (term == null) {
      throw new IllegalArgumentException("Can't create css term with value: " + value);
    }
    return term;
  }

  @NotNull
  public CssElement createExpressionParameter(@NotNull String value, @Nullable Language language) {
    CssTermList propertyValue = createPropertyValue("expression(" + value + ")", language);
    final CssExpression expression = PsiTreeUtil.findChildOfType(propertyValue, CssExpression.class);
    final CssElement expressionParameter = expression != null ? expression.getValueNode() : null;
    if (expressionParameter == null) {
      throw new IllegalArgumentException("Can't create css expression with value: " + value);
    }
    return expressionParameter;
  }

  @NotNull
  public CssPseudoSelector createPseudoSelector(String name, @Nullable Language language) {
    return createPseudoSelector(name, language, false);
  }

  @NotNull
  public CssPseudoSelector createPseudoSelector(String name, @Nullable Language language, boolean isEscaped) {
    assert name.startsWith(":");
    String selector = isEscaped ? "\\" + name : name;
    CssRuleset ruleset = createRuleset(createRulesetText(selector, new StringBuilder(), language).toString(), language);
    final CssPseudoSelector pseudoSelector = PsiTreeUtil.findChildOfType(ruleset, CssPseudoSelector.class);
    if (pseudoSelector == null) {
      throw new IllegalArgumentException("Can't create css pseudo selector from text: " + name);
    }
    return pseudoSelector;
  }

  @NotNull
  public CssSelectorList createSelectorList(@NotNull String text, @Nullable Language language) {
    CssRuleset ruleset = createRuleset(createRulesetText(text, new StringBuilder(), language).toString(), language);
    final CssSelectorList selectorList = PsiTreeUtil.findChildOfType(ruleset, CssSelectorList.class);
    if (selectorList == null) {
      throw new IllegalArgumentException("Can't create css selector list from text: " + text);
    }
    return selectorList;
  }

  @NotNull
  public CssSimpleSelector createSimpleSelector(@NotNull String simpleSelectorText, @Nullable Language language) {
    CssRuleset ruleset = createRuleset(simpleSelectorText + " {}", language);
    final CssSimpleSelector simpleSelector = PsiTreeUtil.findChildOfType(ruleset, CssSimpleSelector.class);
    if (simpleSelector == null) {
      throw new IllegalArgumentException("Can't create css simple selector from text: " + simpleSelectorText);
    }
    return simpleSelector;
  }

  @NotNull
  public CssImportList createImportList(@Nullable Language language, String... imports) {
    String text = StringUtil.join(imports, CssPsiUtil.getDeclarationsTerminator(language));
    final CssStylesheet stylesheet = createStylesheet(text, language);
    final CssImportList importList = stylesheet.getImportList();
    if (importList == null) {
      throw new IllegalArgumentException("Can't create css import list from text: " + text);
    }
    return importList;
  }

  @NotNull
  public PsiElement createToken(@NotNull String text, @Nullable Language language) {
    language = getLanguage(language);
    final PsiFile fileFromText = PsiFileFactory.getInstance(myManager.getProject()).createFileFromText("_dummy_", language, text, false, true);
    final PsiElement leaf = PsiTreeUtil.getDeepestFirst(fileFromText);
    final PsiElement result = leaf instanceof PsiErrorElement ? PsiTreeUtil.nextVisibleLeaf(leaf) : leaf;
    if (result == null) {
      throw new IllegalArgumentException("Can't create css token from text: " + text);
    }
    return result;
  }

  @NotNull
  public CssUri createUri(String string, @Nullable Language language) {
    final CssTermList termList = createPropertyValue("url(" + string + ")", language);
    CssUri cssUri = PsiTreeUtil.findChildOfType(termList, CssUri.class);
    if (cssUri == null) {
      throw new IllegalArgumentException("Can't create css uri with value: " + string);
    }
    return cssUri;
  }

  @NotNull
  private static Language getLanguage(@Nullable Language language) {
    return language != null && language.isKindOf(CSSLanguage.INSTANCE) ? language : CSSLanguage.INSTANCE;
  }

  @NotNull
  private static StringBuilder createRulesetText(@NotNull String selector, @NotNull StringBuilder blockText, @Nullable Language language) {
    boolean identBasedCssLanguage = CssPsiUtil.isIndentBasedCssLanguage(getLanguage(language));
    if (identBasedCssLanguage) {
      if (!blockText.toString().startsWith(" ")) {
        blockText.insert(0, "\n  ");
      }
      else {
        blockText.insert(0, '\n');
      }
    }
    else {
      blockText.insert(0, " {\n");
    }
    blockText.insert(0, selector);
    if (!identBasedCssLanguage) {
      blockText.append('}');
    }
    return blockText;
  }

  @NotNull
  public CssImport createImport(@NotNull String importPath, @Nullable Language language) {
    CssStylesheet stylesheet = createStylesheet("@import '" + importPath + "'", getLanguage(language));
    CssImport cssImport = ContainerUtil.getFirstItem(stylesheet.getImports(false));
    if (cssImport == null) {
      throw new IllegalArgumentException("Can't create css import with import path: " + importPath + " and language: " + language);
    }
    return cssImport;
  }
}
