package com.intellij.psi.css.impl.util.table;

import com.intellij.codeInspection.LocalQuickFix;
import com.intellij.css.util.CssPsiUtil;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.psi.css.*;
import com.intellij.psi.css.descriptor.*;
import com.intellij.psi.css.descriptor.value.CssValueDescriptor;
import com.intellij.psi.css.descriptor.value.CssValueValidator;
import com.intellij.psi.css.descriptor.value.CssValueValidatorStub;
import com.intellij.psi.css.resolve.CssStyleReferenceStub;
import com.intellij.psi.util.*;
import com.intellij.util.ArrayUtil;
import com.intellij.util.Function;
import com.intellij.util.NotNullFunction;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.awt.*;
import java.util.*;
import java.util.List;

/**
 * @author zolotov
 */
public final class CssDescriptorsUtil {

  public static final NotNullFunction<CssElementDescriptor, String> GET_DESCRIPTOR_ID_FUNCTION = new NotNullFunction<CssElementDescriptor, String>() {
    @NotNull
    @Override
    public String fun(CssElementDescriptor descriptor) {
      return descriptor.getId();
    }
  };
  
  private static final Key<ParameterizedCachedValue<CssElementDescriptorProvider, PsiElement>> DESCRIPTOR_PROVIDER_KEY = Key.create("css.descriptor.provider");
  private static final ParameterizedCachedValueProvider<CssElementDescriptorProvider, PsiElement> DESCRIPTOR_PROVIDER_CACHE_VALUE = 
    new ParameterizedCachedValueProvider<CssElementDescriptorProvider, PsiElement>() {
    @Nullable
    @Override
    public CachedValueProvider.Result<CssElementDescriptorProvider> compute(PsiElement param) {
      return CachedValueProvider.Result.create(innerFindDescriptorProvider(param), param);
    }
  };

  private CssDescriptorsUtil() {
  }

  @Nullable
  public static CssElementDescriptorProvider findDescriptorProvider(@Nullable final PsiElement context) {
    final CssStylesheet stylesheet = PsiTreeUtil.getNonStrictParentOfType(context, CssStylesheet.class);
    if (stylesheet != null) {
      return CachedValuesManager.getManager(stylesheet.getProject()).getParameterizedCachedValue(stylesheet, DESCRIPTOR_PROVIDER_KEY,
                                                                                                 DESCRIPTOR_PROVIDER_CACHE_VALUE, false, 
                                                                                                 context);
    }
    return innerFindDescriptorProvider(context);
  }

  private static CssElementDescriptorProvider innerFindDescriptorProvider(PsiElement context) {
    List<CssElementDescriptorProvider> applicableProviders = new LinkedList<CssElementDescriptorProvider>();
    CssElementDescriptorProvider[] providers = CssElementDescriptorProvider.EP_NAME.getExtensions();
    for (CssElementDescriptorProvider provider : providers) {
      if (provider.isMyContext(context)) {
        applicableProviders.add(provider);

        if (!provider.shouldAskOtherProviders(context)) {
          break;
        }
      }
    }
    if (applicableProviders.size() == 1) {
      return ContainerUtil.getFirstItem(applicableProviders);
    }
    return !applicableProviders.isEmpty() ? new CompositeCssElementDescriptorProvider(applicableProviders) : null;
  }

  @NotNull
  public static String[] getSimpleSelectors(@NotNull PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null ? provider.getSimpleSelectors(context) : ArrayUtil.EMPTY_STRING_ARRAY;
  }

  /**
   * @deprecated use {@link this#getPropertyDescriptors(CssDeclaration)} 
   */
  @Nullable
  public static CssPropertyDescriptor getPropertyDescriptor(@NotNull CssDeclaration declaration) {
    return getPropertyDescriptor(declaration.getPropertyName(), declaration);
  }

  @NotNull
  public static Collection<? extends CssPropertyDescriptor> getAllPropertyDescriptors(@Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null ? provider.getAllPropertyDescriptors(context) : Collections.emptyList();
  }

  @NotNull
  public static Collection<? extends CssPseudoSelectorDescriptor> getPseudoSelectorDescriptors(@NotNull String pseudoSelectorName,
                                                                                               @Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null
           ? provider.findPseudoSelectorDescriptors(pseudoSelectorName)
           : Collections.emptyList();
  }

  @NotNull
  public static Collection<? extends CssFunctionDescriptor> getFunctionDescriptors(@NotNull String functionName, @Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null
           ? provider.findFunctionDescriptors(functionName, context)
           : Collections.emptyList();
  }
  
  @NotNull
  public static Collection<? extends CssPropertyDescriptor> getPropertyDescriptors(@NotNull CssDeclaration declaration) {
    return getPropertyDescriptors(declaration.getPropertyName(), declaration);
  }
  
  @NotNull
  public static Collection<? extends CssPropertyDescriptor> getPropertyDescriptors(@NotNull String propertyName, @Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null
           ? provider.findPropertyDescriptors(propertyName, context)
           : Collections.emptyList();
  }
  
  @NotNull
  public static Collection<? extends CssMediaFeatureDescriptor> getMediaFeatureDescriptors(@NotNull String featureName, @Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null
           ? provider.findMediaFeatureDescriptors(featureName, context)
           : Collections.emptyList();
  }
  
  @NotNull
  public static Collection<? extends CssMediaFeatureDescriptor> getAllMediaFeatureDescriptors(@Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null ? provider.getAllMediaFeatureDescriptors(context) : Collections.emptyList();
  }

  /**
   * @deprecated use {@link this#getPropertyDescriptors(String, PsiElement)}
   */
  @Nullable
  public static CssPropertyDescriptor getPropertyDescriptor(@NotNull String propertyName, @Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null ? provider.getPropertyDescriptor(propertyName, context) : null;
  }

  /**
   * @return true if given string is possible simple selector name in given context.
   *
   * Nullable context means that context is simple ruleset.
   */
  public static boolean isPossibleSelector(@NotNull String selector, @NotNull PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider == null || provider.isPossibleSelector(selector, context);
  }

  @NotNull
  public static Collection<LocalQuickFix> getQuickFixesForUnknownSimpleSelector(@NotNull String selectorName, 
                                                                                @NotNull PsiElement context,
                                                                                boolean isOnTheFly) {
    final Set<LocalQuickFix> result = ContainerUtil.newHashSet();
    CssElementDescriptorProvider[] providers = CssElementDescriptorProvider.EP_NAME.getExtensions();
    for (CssElementDescriptorProvider provider : providers) {
      Collections.addAll(result, provider.getQuickFixesForUnknownSimpleSelector(selectorName, context, isOnTheFly));
    }
    return result;
  }

  @NotNull
  public static Collection<LocalQuickFix> getQuickFixesForUnknownProperty(@NotNull String propertyName,
                                                                          @NotNull PsiElement context,
                                                                          boolean isOnTheFly) {
    final Set<LocalQuickFix> result = ContainerUtil.newHashSet();
    CssElementDescriptorProvider[] providers = CssElementDescriptorProvider.EP_NAME.getExtensions();
    for (CssElementDescriptorProvider provider : providers) {
      Collections.addAll(result, provider.getQuickFixesForUnknownProperty(propertyName, context, isOnTheFly));
    }
    return result;
  }


  /**
   * Retrieves list of pseudo-classes and pseudo-elements descriptors, that allowed in given context.
   * Uses in completion providers.
   *
   * @param context Given context. If {@code null} then all possible variants will be returned.
   */
  @NotNull
  public static Collection<? extends CssPseudoSelectorDescriptor> getAllPseudoSelectorDescriptors(@Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null ? provider.getAllPseudoSelectorDescriptors(context) : Collections.<CssPseudoClassDescriptor>emptyList();
  }

  @NotNull
  public static String getCanonicalPropertyName(@NotNull CssDeclaration declaration) {
    String propertyName = declaration.getPropertyName();
    CssPropertyDescriptor descriptor = getPropertyDescriptor(propertyName, declaration);
    return descriptor != null ? descriptor.toCanonicalName(propertyName) : propertyName;
  }

  @Nullable
  public static <T extends CssElementDescriptor> T getDescriptorFromLatestSpec(Collection<T> descriptors) {
    T result = null;
    for (T descriptor : descriptors) {
      if (result == null || descriptor.getCssVersion().value() >= result.getCssVersion().value()) {
        result = descriptor;
      }
    }
    return result;
  }

  @NotNull
  public static String[] extractDescriptorsIdsAsArray(Collection<? extends CssElementDescriptor> descriptors) {
    return ContainerUtil.map2Array(descriptors, String.class, GET_DESCRIPTOR_ID_FUNCTION);
  }

  @NotNull
  public static Set<String> extractDescriptorsIds(Collection<? extends CssElementDescriptor> descriptors) {
    return ContainerUtil.map2LinkedSet(descriptors, GET_DESCRIPTOR_ID_FUNCTION);
  }

  @NotNull
  public static Set<String> extractDocumentations(@NotNull Collection<? extends CssElementDescriptor> descriptors, @Nullable final PsiElement context) {
    return ContainerUtil.map2LinkedSet(descriptors, new Function<CssElementDescriptor, String>() {
      @Override
      public String fun(CssElementDescriptor input) {
        return input.getDocumentationString(context);
      }
    });
  }
  
  @NotNull
  public static <T extends CssMediaGroupAwareDescriptor> Collection<T> filterDescriptorsByMediaType(@NotNull Collection<T> descriptors,
                                                                                                    @Nullable PsiElement context) {
    final Set<CssMediaType> allowedMediaTypes = CssPsiUtil.getAllowedMediaTypesInContext(context);
    if (allowedMediaTypes.contains(CssMediaType.ALL)) {
      return descriptors;
    }
    
    boolean insidePageRule = false;
    CssAtRule atRule = PsiTreeUtil.getNonStrictParentOfType(context, CssAtRule.class);
    while (atRule != null) {
      if (atRule.getType() == CssContextType.PAGE || atRule.getType() == CssContextType.PAGE_MARGIN) {
        insidePageRule = true;
        break;
      }
      atRule = PsiTreeUtil.getParentOfType(atRule, CssAtRule.class);
    }

    final boolean finalInsidePageRule = insidePageRule;
    return ContainerUtil.filter(descriptors, new Condition<T>() {
      @Override
      public boolean value(T t) {
        for (CssMediaType mediaType : allowedMediaTypes) {
          CssMediaGroup[] propertyMediaGroups = t.getMediaGroups();
          for (CssMediaGroup propertyMediaGroup : propertyMediaGroups) {
            if (propertyMediaGroup == CssMediaGroup.ALL
                || finalInsidePageRule && propertyMediaGroup == CssMediaGroup.PAGED
                || ArrayUtil.contains(propertyMediaGroup, (Object[])mediaType.getSupportedGroups())) {
              return true;
            }
          }
        }
        return false;
      }
    });
  }

  @NotNull
  public static <T extends CssElementDescriptor> Collection<T> filterDescriptorsByContext(@NotNull Collection<T> descriptors,
                                                                                          @Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    final CssContextType ruleType = provider != null ? provider.getCssContextType(context) : CssContextType.ANY;
    
    if (ruleType == CssContextType.ANY) {
      return descriptors;
    }
    return ContainerUtil.filter(descriptors, new Condition<T>() {
      @Override
      public boolean value(T input) {
        return input.isAllowedInContextType(ruleType);
      }
    });
  }

  public static Collection<CssPseudoSelectorDescriptor> filterPseudoSelectorDescriptorsByColonPrefix(Collection<? extends CssPseudoSelectorDescriptor> descriptors,
                                                                                                     final int prefixLength) {
    return ContainerUtil.filter(descriptors, new Condition<CssPseudoSelectorDescriptor>() {
      @Override
      public boolean value(CssPseudoSelectorDescriptor input) {
        return input.getColonPrefixLength() == prefixLength;
      }
    });
  }

  @NotNull
  public static <T extends CssElementDescriptor> Collection<T> sortDescriptors(@NotNull Collection<T> descriptors) {
    if (descriptors.size() <= 1) {
        return descriptors;
    }
    
    TreeSet<T> result = ContainerUtil.newTreeSet(new CssVersionDescriptorComparator());
    result.addAll(descriptors);
    return result;
  }

  private static class CompositeCssElementDescriptorProvider extends CssElementDescriptorProvider {
    private final List<CssElementDescriptorProvider> myProviders;

    public CompositeCssElementDescriptorProvider(List<CssElementDescriptorProvider> providers) {
      myProviders = providers;
    }

    @Override
    public boolean isMyContext(@Nullable PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        if (provider.isMyContext(context)) {
          return true;
        }
      }
      return false;
    }

    @Nullable
    @Override
    public CssPropertyDescriptor getPropertyDescriptor(@NotNull String propertyName, @Nullable PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        final CssPropertyDescriptor descriptor = provider.getPropertyDescriptor(propertyName, context);
        if (descriptor != null) {
          return descriptor;
        }
      }
      return null;
    }

    @NotNull
    @Override
    public Collection<? extends CssPseudoSelectorDescriptor> findPseudoSelectorDescriptors(@NotNull String name) {
      Set<CssPseudoSelectorDescriptor> result = ContainerUtil.newHashSet();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.findPseudoSelectorDescriptors(name));
      }
      return result;
    }

    @NotNull
    @Override
    public Collection<? extends CssValueDescriptor> getNamedValueDescriptors(@NotNull String name, @Nullable CssValueDescriptor parent) {
      Set<CssValueDescriptor> result = ContainerUtil.newHashSet();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.getNamedValueDescriptors(name, parent));
      }
      return result;
    }

    @NotNull
    @Override
    public Collection<? extends CssPropertyDescriptor> findPropertyDescriptors(@NotNull String propertyName, PsiElement context) {
      Set<CssPropertyDescriptor> result = ContainerUtil.newHashSet();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.findPropertyDescriptors(propertyName, context));
      }
      return result;
    }

    @NotNull
    @Override
    public Collection<? extends CssFunctionDescriptor> findFunctionDescriptors(@NotNull String functionName, @Nullable PsiElement context) {
      Set<CssFunctionDescriptor> result = ContainerUtil.newHashSet();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.findFunctionDescriptors(functionName, context));
      }
      return result;
    }

    @NotNull
    @Override
    public Collection<? extends CssMediaFeatureDescriptor> findMediaFeatureDescriptors(@NotNull String mediaFeatureName, @Nullable PsiElement context) {
      Set<CssMediaFeatureDescriptor> result = ContainerUtil.newHashSet();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.findMediaFeatureDescriptors(mediaFeatureName, context));
      }
      return result;
    }

    @Override
    public boolean isPossibleSelector(@NotNull String selector, @NotNull PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        if (provider.isPossibleSelector(selector, context)) {
          return true;
        }
      }
      return false;
    }

    @Override
    public boolean isPossiblePseudoSelector(@NotNull String selectorName, @Nullable PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        if (provider.isPossiblePseudoSelector(selectorName, context)) {
          return true;
        }
      }
      return false;
    }

    @NotNull
    @Override
    public Collection<? extends CssPseudoSelectorDescriptor> getAllPseudoSelectorDescriptors(@Nullable PsiElement context) {
      final Set<CssPseudoSelectorDescriptor> result = ContainerUtil.newHashSet();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.getAllPseudoSelectorDescriptors(context));
      }
      return result;
    }

    @NotNull
    @Override
    public Collection<? extends CssPropertyDescriptor> getAllPropertyDescriptors(@Nullable PsiElement context) {
      Set<CssPropertyDescriptor> result = ContainerUtil.newHashSet();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.getAllPropertyDescriptors(context));
      }
      return result;
    }

    @NotNull
    @Override
    public Collection<? extends CssMediaFeatureDescriptor> getAllMediaFeatureDescriptors(@Nullable PsiElement context) {
      Set<CssMediaFeatureDescriptor> result = ContainerUtil.newHashSet();      
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.getAllMediaFeatureDescriptors(context));
      }
      return result;
    }

    @NotNull
    @Override
    public String[] getSimpleSelectors(@NotNull PsiElement context) {
      final Set<String> result = ContainerUtil.newHashSet();
      for (CssElementDescriptorProvider provider : myProviders) {
        Collections.addAll(result, provider.getSimpleSelectors(context));
      }
      return ArrayUtil.toStringArray(result);
    }

    @NotNull
    @Override
    public PsiElement[] getDeclarationsForSimpleSelector(@NotNull CssSimpleSelector selector) {
      for (CssElementDescriptorProvider provider : myProviders) {
        final PsiElement[] declarations = provider.getDeclarationsForSimpleSelector(selector);
        if (declarations.length != 0) {
          return declarations;
        }
      }
      return PsiElement.EMPTY_ARRAY;
    }

    @Override
    public boolean providesClassicCss() {
      for (CssElementDescriptorProvider provider : myProviders) {
        if (!provider.providesClassicCss()) {
          return false;
        }
      }
      return true;
    }

    @Nullable
    @Override
    public PsiElement getDocumentationElementForSelector(@NotNull String selectorName, @Nullable PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        final PsiElement element = provider.getDocumentationElementForSelector(selectorName, context);
        if (element != null) {
          return element;
        }
      }
      return null;
    }

    @Nullable
    @Override
    public String generateDocForSelector(@NotNull String selectorName, @Nullable PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        final String doc = provider.generateDocForSelector(selectorName, context);
        if (doc != null) {
          return doc;
        }
      }
      return null;
    }

    @Nullable
    @Override
    public Color getColorByValue(@NotNull String value) {
      for (CssElementDescriptorProvider provider : myProviders) {
        final Color color = provider.getColorByValue(value);
        if (color != null) {
          return color;
        }
      }
      return null;
    }

    @Override
    public boolean isColorTerm(@NotNull CssTerm term) {
      for (CssElementDescriptorProvider provider : myProviders) {
        if (provider.isColorTerm(term)) {
          return true;
        }
      }
      return false;
    }

    @NotNull
    @Override
    public LocalQuickFix[] getQuickFixesForUnknownProperty(@NotNull String propertyName, @NotNull PsiElement context, boolean isOnTheFly) {
      final Set<LocalQuickFix> quickFixes = ContainerUtil.newHashSet();
      for (CssElementDescriptorProvider provider : myProviders) {
        Collections.addAll(quickFixes, provider.getQuickFixesForUnknownProperty(propertyName, context, isOnTheFly));
      }
      return quickFixes.toArray(new LocalQuickFix[quickFixes.size()]);
    }

    @NotNull
    @Override
    public LocalQuickFix[] getQuickFixesForUnknownSimpleSelector(@NotNull String selectorName,
                                                                 @NotNull PsiElement context,
                                                                 boolean isOnTheFly) {
      final Set<LocalQuickFix> quickFixes = ContainerUtil.newHashSet();
      for (CssElementDescriptorProvider provider : myProviders) {
        Collections.addAll(quickFixes, provider.getQuickFixesForUnknownSimpleSelector(selectorName, context, isOnTheFly));
      }
      return quickFixes.toArray(new LocalQuickFix[quickFixes.size()]);
    }

    @Override
    public boolean isColorTermsSupported() {
      for (CssElementDescriptorProvider provider : myProviders) {
        if (!provider.isColorTermsSupported()) {
          return false;
        }
      }
      return true;
    }

    @Override
    public CssContextType getCssContextType(@Nullable PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        final CssContextType ruleType = provider.getCssContextType(context);
        if (ruleType != CssContextType.ANY) {
          return ruleType;
        }
      }
      return CssContextType.ANY;
    }

    @NotNull
    @Override
    public PsiReference getStyleReference(PsiElement element, int start, int end, boolean caseSensitive) {
      for (CssElementDescriptorProvider provider : myProviders) {
        PsiReference reference = provider.getStyleReference(element, start, end, caseSensitive);
        if (!(reference instanceof CssStyleReferenceStub)) {
          return reference;
        }
      }
      return new CssStyleReferenceStub(element, TextRange.create(start, end));
    }

    @NotNull
    @Override
    public CssValueValidator getValueValidator() {
      for (CssElementDescriptorProvider provider : myProviders) {
        CssValueValidator validator = provider.getValueValidator();
        if (!(validator instanceof CssValueValidatorStub)) {
          return validator;
        }
      }
      return super.getValueValidator();
    }
  }
}
