/*
 * Copyright 2000-2012 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database.util;

import com.intellij.database.model.DatabaseSystem;
import com.intellij.database.psi.DbDataSource;
import com.intellij.openapi.util.Comparing;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.ui.CollectionComboBoxModel;
import com.intellij.ui.JBColor;
import com.intellij.ui.ListCellRendererWrapper;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.ui.UIUtil;
import org.jetbrains.annotations.NotNull;

import javax.swing.*;
import java.awt.*;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;

/**
 * @author Gregory.Shrago
 */
public class DbUIUtil {
  private DbUIUtil() {
  }

  public static void configureDataSourceComboBox(JComboBox comboBox, Iterable<DbDataSource> dataSources, boolean allowNull) {
    ArrayList<DbDataSource> copy = ContainerUtil.newArrayList(dataSources);
    Collections.sort(copy, new Comparator<DatabaseSystem>() {
      public int compare(final DatabaseSystem o1, final DatabaseSystem o2) {
        return Comparing.compare(StringUtil.toLowerCase(o1.getName()), StringUtil.toLowerCase(o2.getName()));
      }
    });
    if (allowNull) copy.add(0, null);
    comboBox.setModel(new CollectionComboBoxModel<DbDataSource>(copy));
    comboBox.setRenderer(new ListCellRendererWrapper<DatabaseSystem>() {
      @Override
      public void customize(JList list, DatabaseSystem element, int index, boolean selected, boolean hasFocus) {
        setIcon(element instanceof DbDataSource ? ((DbDataSource)element).getIcon() : null);
        final String name = element == null ? "<none>" : element.getName();
        setText(StringUtil.notNullize(name));
      }
    });
    comboBox.setEditable(false);
  }

  public static void invokeLaterIfNeeded(@NotNull Runnable runnable) {
    UIUtil.invokeLaterIfNeeded(runnable);
  }

  /** @noinspection SSBasedInspection*/
  public static void invokeLater(@NotNull Runnable runnable) {
    SwingUtilities.invokeLater(runnable);
  }

  public static Color softHighlightOf(Color c1) {
    if(c1 == null) return null;
    int i = 0x10;
    return new JBColor(new Color(Math.max(0, c1.getRed() - i), Math.max(0, c1.getGreen() - i), Math.max(0, c1.getBlue() - i)),
                       new Color(Math.min(255, c1.getRed() + i), Math.min(255, c1.getGreen() + i), Math.min(255, c1.getBlue() + i)));
  }
}
