/*
 * Copyright 2000-2013 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database.util;

import com.intellij.database.DatabaseFamilyId;
import com.intellij.database.dataSource.DataSource;
import com.intellij.database.model.*;
import com.intellij.database.psi.DbDataSource;
import com.intellij.database.psi.DbElement;
import com.intellij.database.psi.DbPsiFacade;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.text.StringUtil;
import gnu.trove.THashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.TestOnly;

import java.util.Set;
import java.util.regex.Pattern;

/**
 * @author gregsh
 */
public class LoaderContext {
  private final DataSource myDataSource;
  private final DatabaseFamilyId myFamilyId;
  private ErrorHandler myErrorHandler = new ErrorHandler();

  private final THashSet<Object> mySelection = new THashSet<Object>();
  private Object mySelectedElement;
  private Runnable myUpdateUiRunnable;
  private Pattern myProcedurePattern;

  public static LoaderContext loadAll(DataSource dataSource) {
    return new LoaderContext(dataSource).includeAllSchemas().includeTables().includeProcedures();
  }

  public static LoaderContext loadOne(DataSource dataSource, Object o) {
    return new LoaderContext(dataSource).include(o);
  }

  public LoaderContext(DataSource dataSource) {
    myDataSource = dataSource;
    myFamilyId = DatabaseFamilyId.forDataSource(myDataSource);
  }

  public DataSource getDataSource() {
    return myDataSource;
  }

  public DatabaseFamilyId getFamilyId() {
    return myFamilyId;
  }

  public Set<Object> getSelection() {
    return mySelection;
  }

  public LoaderContext setErrorHandler(ErrorHandler handler) {
    myErrorHandler = handler;
    return this;
  }

  public LoaderContext include(@Nullable DasObject cur, String newName, boolean focusInView) {
    include(cur);
    return this;
  }

  public LoaderContext include(@Nullable Object o) {
    if (o == null) return this;

    if (o instanceof DasNamespace) {
      includeNamespace((DasNamespace)o);
    }
    else {
      mySelection.add(o instanceof DbElement ? ((DbElement)o).getDelegate() : o);
    }

    return this;
  }

  /**
   * @deprecated use {@link #includeNamespace(DasNamespace)} instead.
   */
  @Deprecated
  public LoaderContext includeSchema(String catalog, String schema) {
    mySelection.add(getPattern(catalog, schema));
    return this;
  }

  public LoaderContext includeNamespace(DasNamespace namespace) {
    final String pattern;

    final ObjectKind namespaceKind = namespace.getKind();
    if (namespaceKind == ObjectKind.DATABASE) {
      pattern = getPattern(namespace.getName(), null);
    }
    else if (namespaceKind == ObjectKind.SCHEMA) {
      final DasObject parent = namespace.getDbParent();
      if (myFamilyId.isOracle()) {
        // TODO remove this ugly hack when refactor schema patterns
        pattern = getPattern(null, namespace.getName());
      }
      else if (parent != null && parent.getKind() == ObjectKind.DATABASE) {
        pattern = getPattern(parent.getName(), namespace.getName());
      }
      else if (myFamilyId.isMysql()) {
        // TODO remove this ugly hack when refactor schema patterns
        pattern = getPattern(namespace.getName(), null);
      }
      else {
        pattern = getPattern(null, namespace.getName());
      }
    }
    else {
      throw new IllegalArgumentException("Strange kind of the given namespace: " + namespaceKind.name());
    }

    mySelection.add(pattern);
    return this;
  }

  public LoaderContext includeAllSchemas() {
    mySelection.add(getPattern(null, null));
    return this;
  }

  public LoaderContext includeAllEntities() {
    return includeTables().includeProcedures();
       // TODO what's about other kinds - sequences, views, packages, etc.?
  }

  public LoaderContext includeTables() {
    return include(DasTable.class);
  }

  public LoaderContext includeProcedures() {
    return include(DasRoutine.class);
  }

  public LoaderContext setUpdateUi(@NotNull final Project project, @Nullable final Runnable continuation) {
    myUpdateUiRunnable = new Runnable() {
      @Override
      public void run() {
        if (!project.isOpen()) return;
        DbDataSource dataSourceElement = DbPsiFacade.getInstance(project).findDataSource(myDataSource.getUniqueId());
        if (dataSourceElement != null) {
          DbPsiFacade.getInstance(project).clearCaches(dataSourceElement);
          dataSourceElement.getDbManager().fireDataSourceUpdated(dataSourceElement);
        }
        if (continuation != null) continuation.run();
      }
    };
    return this;
  }

  public boolean loadNothing() {
    return mySelection.isEmpty();
  }

  public boolean load(Object o) {
    return mySelection.contains(o);
  }

  public boolean load(String catalog, String schema) {
    return mySelection.contains(getPattern(catalog, schema)) ||
           mySelection.contains(getPattern(catalog, null)) ||
           mySelection.contains(getPattern(null, null));
  }

  public boolean loadTables() {
    return load(DasTable.class);
  }

  public boolean loadProcedures() {
    return load(DasRoutine.class);
  }

  public ErrorHandler getErrorHandler() {
    return myErrorHandler;
  }

  @Nullable
  public Runnable getUpdateUiRunnable() {
    return myUpdateUiRunnable;
  }

  public Object getElementToFocus() {
    return mySelectedElement;
  }

  public void setElementToFocus(@Nullable Object selectedElement) {
    mySelectedElement = selectedElement;
  }

  @NotNull
  public static String getPattern(@Nullable String catalog, @Nullable String schema) {
    if (StringUtil.isNotEmpty(catalog)) {
      if (StringUtil.isNotEmpty(schema)) {
        return catalog + "." + schema;
      }
      else {
        return catalog + ".*";
      }
    }
    else if (StringUtil.isNotEmpty(schema)) {
      return "*." + schema;
    }
    else {
      return "*";
    }
  }

  @Nullable
  public Pattern getProcedurePattern() {
    return myProcedurePattern;
  }

  @TestOnly
  public LoaderContext setProcedurePattern(@Nullable Pattern procedurePattern) {
    myProcedurePattern = procedurePattern;
    return this;
  }
}
