/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.javaee.web;

import com.intellij.facet.FacetTypeId;
import com.intellij.javaee.application.facet.JavaeeApplicationFacet;
import com.intellij.javaee.artifact.JavaeeArtifactUtil;
import com.intellij.javaee.context.FacetContextProvider;
import com.intellij.javaee.context.WebModuleContextProvider;
import com.intellij.javaee.deployment.DeploymentModel;
import com.intellij.javaee.facet.JavaeeFacet;
import com.intellij.javaee.web.artifact.WebArtifactUtil;
import com.intellij.javaee.web.facet.WebFacet;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.io.FileUtil;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.packaging.artifacts.Artifact;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.util.Collection;

/**
 * Created by IntelliJ IDEA.
 * User: michael.golubev
 */
public class WebFacetContextProvider implements FacetContextProvider {

  @Override
  public FacetTypeId<? extends JavaeeFacet> getFacetId() {
    return WebFacet.ID;
  }

  @Override
  public String getDeploymentContext(@NotNull WebModuleContextProvider webModuleContextProvider,
                                     @NotNull DeploymentModel deploymentModel,
                                     @NotNull JavaeeFacet facet) {
    WebFacet webFacet = (WebFacet)facet;

    Artifact artifact = deploymentModel.getArtifact();
    if (artifact != null && JavaeeArtifactUtil.getInstance().isJavaeeApplication(artifact.getArtifactType())) {
      return getModuleContext(webModuleContextProvider, artifact, webFacet);
    }

    String serverSpecificContext = StringUtil.nullize(getContextServerSpecific(deploymentModel, webFacet));
    if (serverSpecificContext != null) {
      return serverSpecificContext;
    }

    File source = deploymentModel.getDeploymentSource().getFile();
    return source == null ? null : FileUtil.getNameWithoutExtension(source);
  }

  protected String getContextServerSpecific(@NotNull DeploymentModel deploymentModel, @NotNull WebFacet facet) {
    return getContextServerSpecific(facet);
  }

  protected String getContextServerSpecific(@NotNull WebFacet facet) {
    return null;
  }

  @Nullable
  public static String getModuleContext(@NotNull WebModuleContextProvider webModuleContextProvider,
                                        @NotNull Artifact earArtifact,
                                        @NotNull final WebFacet webFacet) {
    final Project project = webFacet.getModule().getProject();
    final String moduleWebUri = WebArtifactUtil.getInstance().getModuleWebUri(earArtifact, webFacet);
    if (moduleWebUri == null) {
      return null;
    }

    final Collection<JavaeeApplicationFacet> earFacets
      = JavaeeArtifactUtil.getInstance().getFacetsIncludedInArtifact(project, earArtifact, JavaeeApplicationFacet.ID);
    for (JavaeeApplicationFacet earFacet : earFacets) {
      String contextRoot = webModuleContextProvider.getContext(earFacet, moduleWebUri);
      if (contextRoot != null) {
        return contextRoot;
      }
    }
    return null;
  }
}
