/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.aliasFor;

import com.intellij.codeInsight.AnnotationUtil;
import com.intellij.jam.JamService;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.spring.model.jam.utils.JamAnnotationTypeUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;

/**
 * @since 16
 */
public class SpringAliasForUtils {

  @Nullable
  public static SpringAliasFor findAliasFor(@NotNull PsiElement context,
                                            @Nullable String toSearchInAnnotation,
                                            @NotNull String aliasedClassName,
                                            @NotNull String attrName) {
    if (toSearchInAnnotation == null) return null;

    final JavaPsiFacade javaPsiFacade = JavaPsiFacade.getInstance(context.getProject());
    PsiClass toSearchInClass = javaPsiFacade.findClass(toSearchInAnnotation, context.getResolveScope());
    if (toSearchInClass == null || !toSearchInClass.isAnnotationType()) {
      return null;
    }

    PsiClass annotationClass = javaPsiFacade.findClass(aliasedClassName, context.getResolveScope());
    if (annotationClass == null || !annotationClass.isAnnotationType()) {
      return null;
    }

    for (SpringAliasFor aliasFor : getAliasForAttributes(toSearchInClass)) {
      if (attrName.equals(aliasFor.getAttributeName()) && annotationClass.equals(aliasFor.getAnnotationClass())) {
        return aliasFor;
      }
    }
    return null;
  }

  @NotNull
  private static List<SpringAliasFor> getAliasForAttributes(@NotNull PsiClass psiClass) {
    return JamService.getJamService(psiClass.getProject())
      .getAnnotatedMembersList(psiClass, SpringAliasFor.SEM_KEY,
                               false, true, false, false);
  }

  @Nullable
  public static PsiAnnotation findDefiningMetaAnnotation(@NotNull PsiElement context,
                                                         @NotNull String customAnnotationFqn,
                                                         @NotNull String baseMetaAnnotationFqn) {
    final Module module = ModuleUtilCore.findModuleForPsiElement(context);
    if (module == null) return null;

    PsiClass customAnnoClass = JavaPsiFacade.getInstance(module.getProject())
      .findClass(customAnnotationFqn, context.getResolveScope());

    return findDefiningMetaAnnotation(customAnnoClass, baseMetaAnnotationFqn,
                                      JamAnnotationTypeUtil.getInstance(module).getAnnotationTypesWithChildren(baseMetaAnnotationFqn));
  }

  @Nullable
  public static PsiAnnotation findDefiningMetaAnnotation(@Nullable PsiClass customAnnoClass,
                                                          @NotNull String baseMetaAnnotationFqn,
                                                          @NotNull final Collection<PsiClass> allMetaAnnotations) {
    if (customAnnoClass == null || !customAnnoClass.isAnnotationType()) {
      return null;
    }

    final PsiAnnotation psiAnnotation = AnnotationUtil.findAnnotation(customAnnoClass, true, baseMetaAnnotationFqn);
    if (psiAnnotation != null) {
      return psiAnnotation;
    }

    for (PsiClass customMetaAnnoClass : allMetaAnnotations) {
      final String qualifiedName = customMetaAnnoClass.getQualifiedName();
      if (qualifiedName != null && AnnotationUtil.isAnnotated(customAnnoClass, qualifiedName, true)) {
        return findDefiningMetaAnnotation(customMetaAnnoClass, baseMetaAnnotationFqn, allMetaAnnotations);
      }
    }
    return null;
  }
}
