/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.spring.model.jam.javaConfig;

import com.intellij.jam.JamPomTarget;
import com.intellij.jam.JamSimpleReferenceConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamAttributeMeta;
import com.intellij.jam.reflect.JamMethodMeta;
import com.intellij.jam.reflect.JamStringAttributeMeta;
import com.intellij.pom.PomNamedTarget;
import com.intellij.pom.references.PomService;
import com.intellij.psi.*;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.jam.JamPsiMemberSpringBean;
import com.intellij.util.ArrayUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

public abstract class ContextJavaBean extends SpringJavaBean {
  public static final SemKey<ContextJavaBean> BEAN_JAM_KEY = JamPsiMemberSpringBean.PSI_MEMBER_SPRING_BEAN_JAM_KEY.subKey("ContextJavaBean");
  public static final JamMethodMeta<ContextJavaBean> METHOD_META = new JamMethodMeta<ContextJavaBean>(null, ContextJavaBean.class, BEAN_JAM_KEY);

  // @Bean annotation meta
  private static final JamStringAttributeMeta.Collection<String> NAME_VALUE_META = JamAttributeMeta.collectionString("name");
  private static final JamStringAttributeMeta.Single<PsiMethod> INIT_METHOD_META = JamAttributeMeta.singleString("initMethod", new PsiMethodReferenceConverter());
  private static final JamStringAttributeMeta.Single<PsiMethod> DESTROY_METHOD_META = JamAttributeMeta.singleString("destroyMethod", new PsiMethodReferenceConverter());
  public static final JamAnnotationMeta ANNOTATION_META = new JamAnnotationMeta(SpringAnnotationsConstants.JAVA_SPRING_BEAN).addAttribute(NAME_VALUE_META).addAttribute(INIT_METHOD_META).addAttribute(DESTROY_METHOD_META);

  static {
    METHOD_META.addAnnotation(ANNOTATION_META);
  }

  public PsiAnnotation getPsiAnnotation() {
    return ANNOTATION_META.getAnnotation(getPsiElement());
  }

  @Override
  public String getBeanName() {
    List<JamStringAttributeElement<String>> elements = getNamedAttributeValue();
    if (elements.size() > 0) {
      return elements.get(0).getValue();
    }

    return super.getBeanName();
  }

  @NotNull
  public JamStringAttributeElement<PsiMethod> getInitMethodAttributeElement() {
    return ANNOTATION_META.getAttribute(getPsiElement(), INIT_METHOD_META);
  }

  @NotNull
  public JamStringAttributeElement<PsiMethod> getDestroyMethodAttributeElement() {
    return ANNOTATION_META.getAttribute(getPsiElement(), DESTROY_METHOD_META);
  }

  @NotNull
  public String[] getAliases() {
    // @Bean "name" attribute: The name of this bean, or if plural, aliases for this bean.
    List<JamStringAttributeElement<String>> elements = getNamedAttributeValue();

    if (elements.size() < 2) return ArrayUtil.EMPTY_STRING_ARRAY;

    List<String> aliases = getStringNames(elements);

    return ArrayUtil.toStringArray(aliases);
  }

  public List<PomNamedTarget> getPomTargets() {
    List<PomNamedTarget> pomTargets = new LinkedList<PomNamedTarget>();

    List<JamStringAttributeElement<String>> elements = getNamedAttributeValue();
    if (!elements.isEmpty()) {
      for (JamStringAttributeElement<String> attributeElement : elements) {
        pomTargets.add(new JamPomTarget(this, attributeElement));
      }
    }
    else {
      pomTargets.add(getPsiElement());
    }

    return pomTargets;
  }

  protected List<JamStringAttributeElement<String>> getNamedAttributeValue() {
    return ANNOTATION_META.getAttribute(getPsiElement(), NAME_VALUE_META);
  }

  @Override
  public PsiNamedElement getIdentifyingPsiElement() {
    final PsiTarget psiTarget = getPsiTarget();
    if (psiTarget != null) {
      final PsiElement psiElement = PomService.convertToPsi(getPsiManager().getProject(), psiTarget);
      if (psiElement instanceof PsiNamedElement) return (PsiNamedElement)psiElement;
    }
    return super.getIdentifyingPsiElement();
  }

  @Nullable
  public PsiTarget getPsiTarget() {
    final List<JamStringAttributeElement<String>> namedAttributeValue = getNamedAttributeValue();
    if (namedAttributeValue == null || namedAttributeValue.size() == 0) {
      return null;
    }

    return new JamPomTarget(this, namedAttributeValue.get(0));
  }

  private static class PsiMethodReferenceConverter extends JamSimpleReferenceConverter<PsiMethod> {
    public PsiMethod fromString(@Nullable String s, JamStringAttributeElement<PsiMethod> context) {
      for (PsiMethod psiMethod : getAppropriateMethods(context)) {
        if (psiMethod.getName().equals(s)) {
          return psiMethod;
        }
      }
      return null;
    }

    private static List<PsiMethod> getAppropriateMethods(JamStringAttributeElement<PsiMethod> context) {
      List<PsiMethod> methods = new ArrayList<PsiMethod>();
      PsiMethod method = PsiTreeUtil.getParentOfType(context.getPsiElement(), PsiMethod.class);

      if (method != null) {
        PsiType type = method.getReturnType();
        if (type instanceof PsiClassType) {
          final PsiClass psiClass = ((PsiClassType)type).resolve();
          if (psiClass != null) {
            for (PsiMethod psiMethod : psiClass.getAllMethods()) {
              if (!psiMethod.isConstructor() && psiMethod.getParameterList().getParametersCount() == 0) {
                methods.add(psiMethod);
              }
            }
          }
        }
      }
      return methods;
    }

    @Override
    public Collection<PsiMethod> getVariants(JamStringAttributeElement<PsiMethod> context) {
      List<PsiMethod> methods = new ArrayList<PsiMethod>();
      for (PsiMethod method : getAppropriateMethods(context)) {
        final PsiClass containingClass = method.getContainingClass();
        if (containingClass != null) {
          if (!CommonClassNames.JAVA_LANG_OBJECT.equals(containingClass.getQualifiedName())) {
            methods.add(method);
          }
        }
      }
      return methods;
    }
  }

}
