/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.values.converters.resources;

import com.intellij.openapi.extensions.Extensions;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.Conditions;
import com.intellij.openapi.util.Pair;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.util.InheritanceUtil;
import com.intellij.spring.constants.SpringConstants;
import com.intellij.spring.model.utils.SpringReferenceUtils;
import com.intellij.spring.model.utils.resources.SpringResourcesBuilder;
import com.intellij.spring.model.utils.resources.SpringResourcesUtil;
import com.intellij.util.Function;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.ConvertContext;
import com.intellij.util.xml.Converter;
import com.intellij.util.xml.CustomReferenceConverter;
import com.intellij.util.xml.GenericDomValue;
import com.intellij.util.xml.impl.ConvertContextFactory;
import gnu.trove.THashSet;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

/**
 * Use {@link SpringResourceTypeProvider} to filter specific files (e.g. by extension, XSD).
 */
public class ResourceValueConverter extends Converter<Set<PsiFileSystemItem>> implements CustomReferenceConverter {

  public Set<PsiFileSystemItem> fromString(@Nullable @NonNls String s, final ConvertContext context) {
    final GenericDomValue domValue = (GenericDomValue)context.getInvocationElement();
    if (StringUtil.isEmpty(s)) {
      return Collections.emptySet();
    }

    return addResourceFilesFrom(domValue, new THashSet<PsiFileSystemItem>(), getFilter(), this);
  }

  private static <V extends PsiFileSystemItem> Set<V> addResourceFilesFrom(final @NotNull GenericDomValue element,
                                                                           final Set<V> result,
                                                                           final Condition<PsiFileSystemItem> filter,
                                                                           Converter converter) {
    if (converter instanceof CustomReferenceConverter) {
      @SuppressWarnings("unchecked")
      final PsiReference[] references =
        ((CustomReferenceConverter)converter)
          .createReferences(element, element.getXmlElement(), ConvertContextFactory.createConvertContext(element));
      result.addAll(SpringResourcesUtil.getInstance().<V>getResourceItems(references, filter));
      return result;
    }
    return result;
  }

  public String toString(@Nullable Set<PsiFileSystemItem> o, final ConvertContext context) {
    return null;
  }

  @NotNull
  public PsiReference[] createReferences(final GenericDomValue genericDomValue, final PsiElement element, final ConvertContext context) {
    final int startInElement = ElementManipulators.getOffsetInElement(element);
    final Condition<PsiFileSystemItem> filterCondition = getFilter(genericDomValue);
    final Function<PsiFile, Collection<PsiFileSystemItem>> customDefaultPathEvaluator = getCustomDefaultPathEvaluator(element);

    final List<PsiReference> result = new SmartList<PsiReference>();
    SpringReferenceUtils.processSeparatedString(genericDomValue.getStringValue(), ",", (s, offset) -> {
      final SpringResourcesBuilder builder = SpringResourcesBuilder.create(element, s).
        fromRoot(true).
        offset(offset + startInElement).
        filter(filterCondition).
        endingSlashNotAllowed(true).
        customDefaultPathEvaluator(customDefaultPathEvaluator);
      ContainerUtil.addAll(result, SpringResourcesUtil.getInstance().getReferences(builder));
      return true;
    });
    return result.isEmpty() ? PsiReference.EMPTY_ARRAY : result.toArray(new PsiReference[result.size()]);
  }

  protected Condition<PsiFileSystemItem> getFilter() {
    return Conditions.alwaysTrue();
  }

  @Nullable
  protected Function<PsiFile, Collection<PsiFileSystemItem>> getCustomDefaultPathEvaluator(@SuppressWarnings("UnusedParameters") PsiElement element) {
    return null;
  }

  @NotNull
  protected Condition<PsiFileSystemItem> getFilter(@NotNull GenericDomValue genericDomValue) {
    for (SpringResourceTypeProvider provider : Extensions.getExtensions(SpringResourceTypeProvider.EP_NAME)) {
      Condition<PsiFileSystemItem> filter = provider.getResourceFilter(genericDomValue);
      if (filter != null) return filter;
    }
    return getFilter();
  }


  public static class ResourceValueConverterCondition implements Condition<Pair<PsiType, GenericDomValue>> {

    public boolean value(Pair<PsiType, GenericDomValue> pair) {
      PsiType psiType = pair.getFirst();
      if (psiType instanceof PsiArrayType) {
        psiType = ((PsiArrayType)psiType).getComponentType();
      }
      if (!(psiType instanceof PsiClassType)) {
        return false;
      }

      final String psiTypeText = psiType.getCanonicalText();
      if (CommonClassNames.JAVA_LANG_STRING.equals(psiTypeText)) {
        return false;
      }

      return SpringConstants.IO_RESOURCE.equals(psiTypeText) ||
             CommonClassNames.JAVA_IO_FILE.equals(psiTypeText) ||
             InheritanceUtil.isInheritor(psiType, "java.io.InputStream");
    }
  }
}
