/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.sql.util;

import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.tree.IElementType;
import com.intellij.sql.psi.SqlCompositeElementType;
import com.intellij.sql.psi.impl.SqlKeywordTokenType;
import com.intellij.sql.psi.impl.SqlTokenType;
import com.intellij.util.NotNullFunction;
import com.intellij.util.NullableFunction;
import com.intellij.util.PairProcessor;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.text.CaseInsensitiveStringHashingStrategy;
import gnu.trove.THashMap;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.Field;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

/**
 * @author Gregory.Shrago
 */
public class SqlTokenRegistry {
  private static final THashMap<String, SqlTokenType> ourTokensMap = newTokenMap();
  private static final THashMap<Class, Map<String, SqlKeywordTokenType>> ourClassTokensMap = ContainerUtil.newTroveMap();
  private static final THashMap<String, IElementType> ourCompositeMap = ContainerUtil.newTroveMap();

  private SqlTokenRegistry() {
  }

  @NotNull
  public static synchronized SqlTokenType getType(@NonNls @NotNull String text) {
    SqlTokenType type = ourTokensMap.get(text);
    if (type == null) {
      ourTokensMap.put(text, type = (Character.isLetter(text.charAt(0))? new SqlKeywordTokenType(text) : new SqlTokenType(text)));
    }
    return type;
  }

  @NotNull
  public static synchronized SqlTokenType getType(@NonNls @NotNull String text, NotNullFunction<String, ? extends SqlTokenType> factory) {
    SqlTokenType type = ourTokensMap.get(text);
    if (type == null) {
      ourTokensMap.put(text, type = factory.fun(text));
    }
    return type;
  }

  @NotNull
  public static synchronized SqlCompositeElementType getCompositeType(@NonNls @NotNull String debugName) {
    SqlCompositeElementType type = (SqlCompositeElementType)ourCompositeMap.get(debugName);
    if (type == null) ourCompositeMap.put(debugName, type = new SqlCompositeElementType(debugName));
    return type;
  }

  @NotNull
  public static synchronized <T extends IElementType> T getCompositeType(@NonNls @NotNull final String debugName, NotNullFunction<String, T> factory) {
    T type = (T)ourCompositeMap.get(debugName);
    if (type == null) ourCompositeMap.put(debugName, type = factory.fun(debugName));
    return type;
  }

  @Nullable
  public static synchronized SqlTokenType findTokenType(@NonNls @NotNull final String text) {
    return ourTokensMap.get(text);
  }

  @Nullable
  public static synchronized IElementType findCompositeType(@NonNls @NotNull final String text) {
    return ourCompositeMap.get(text);
  }

  @Nullable
  public static synchronized IElementType findType(@NonNls @NotNull String text) {
    SqlTokenType token = ourTokensMap.get(text);
    if (token != null) return token;
    return ourCompositeMap.get(text);
  }

  public static void initTypeMap(Class clazz, @Nullable Set<String> exclude) {
    Map<String, SqlKeywordTokenType> map = buildTokenMap(clazz, newTokenMap());
    if (exclude != null) {
      map.keySet().removeAll(exclude);
    }
    setTypeMap(clazz, map, false);
  }

  public static synchronized void addTypeMap(final Class clazz, final Set<String> keywordSet) {
    Map<String, SqlKeywordTokenType> existing = ourClassTokensMap.get(clazz);
    Map<String, SqlKeywordTokenType> map = existing == null ? newTokenMap() : existing;
    for (String s : keywordSet) {
      if (!map.containsKey(s)) {
        String upperCase = StringUtil.toUpperCase(s);
        map.put(upperCase, (SqlKeywordTokenType)getType(upperCase));
      }
    }
    setTypeMap(clazz, map, true);
  }

  private static synchronized void setTypeMap(Class clazz, Map<String, SqlKeywordTokenType> map, boolean force) {
    if (!force && ourClassTokensMap.containsKey(clazz)) return;
    ourClassTokensMap.put(clazz, map);
  }

  @NotNull
  public static NullableFunction<String, IElementType> getTokenProvider(@NotNull final Class clazz) {
    return getSafeMap(clazz)::get;
  }

  public static Set<String> getTokens(final Class clazz) {
    return Collections.unmodifiableSet(getSafeMap(clazz).keySet());
  }

  @NotNull
  private synchronized static Map<String, SqlKeywordTokenType> getSafeMap(final Class clazz) {
    Map<String, SqlKeywordTokenType> result = ourClassTokensMap.get(clazz);
    if (result == null) {
      throw new AssertionError(clazz + " token map not initialized");
    }
    return result;
  }

  private static <T extends IElementType> THashMap<String, T> newTokenMap() {
    return new THashMap<String, T>(CaseInsensitiveStringHashingStrategy.INSTANCE);
  }

  private static Map<String, SqlKeywordTokenType> buildTokenMap(final Class clazz, final Map<String, SqlKeywordTokenType> map) {
    processStaticFields(clazz, (field, value) -> {
      if (value instanceof SqlKeywordTokenType) {
        final SqlKeywordTokenType tokenType = (SqlKeywordTokenType)value;
        map.put(tokenType.toString(), tokenType);
      }
      return true;
    });
    return map;
  }

  public static boolean processStaticFields(final Class clazz, final PairProcessor<Field, Object> processor) {
    for (Field field : clazz.getFields()) {
      try {
        final Object value = field.get(null);
        if (!processor.process(field, value)) return false;
      }
      catch (IllegalAccessException ignored) {
      }
    }
    return true;
  }
}
