/**
 * Entry point for the TypeScript plugin
 */

import {getService} from './service-loader'
import {getSession} from "./ts-session";
import {serverLogger, createLoggerFromEnv} from "./logger-impl";

class TypeScriptLanguagePlugin implements LanguagePlugin {

    private _session:ts.server.Session;

    constructor(state:TypeScriptPluginState) {
        let loggerImpl = createLoggerFromEnv();

        let serviceInfo = getService(state.serverFolderPath);
        let serviceContext:{
            ts:any
        } = serviceInfo.context;

        let serverFilePath = serviceInfo.serverFilePath;
        let tsImpl = serviceContext.ts;

        overrideSysDefaults(tsImpl, serverFilePath);
        this._session = getSession(tsImpl, loggerImpl);
    }

    onMessage(p:string) {
        this._session.onMessage(p);
    }
}

class TypeScriptLanguagePluginFactory implements LanguagePluginFactory {
    create(state:PluginState):LanguagePlugin {
        return new TypeScriptLanguagePlugin(<TypeScriptPluginState>state);
    }
}


function overrideSysDefaults(ts_impl:typeof ts, serverFolderPath:string) {
    const pending:string[] = [];
    let canWrite = true;

    function writeMessage(s:string) {
        if (!canWrite) {
            pending.push(s);
        }
        else {
            canWrite = false;
            process.stdout.write(new Buffer(s, "utf8"), setCanWriteFlagAndWriteMessageIfNecessary);
        }
    }

    function setCanWriteFlagAndWriteMessageIfNecessary() {
        canWrite = true;
        if (pending.length) {
            writeMessage(pending.shift());
        }
    }

    // Override sys.write because fs.writeSync is not reliable on Node 4
    ts_impl.sys.write = (s:string) => writeMessage(s);

    //ts 2.0 compatibility
    (<any>ts_impl.sys).setTimeout = setTimeout;
    (<any>ts_impl.sys).clearTimeout = clearTimeout;

    ts_impl.sys.getExecutingFilePath = () => {
        return serverFolderPath;
    }
}

var typescriptLanguagePluginFactory:TypeScriptLanguagePluginFactory = new TypeScriptLanguagePluginFactory();

export {typescriptLanguagePluginFactory}



