import {serverLogger, isLogEnabled} from "./logger-impl";
import CompletionsRequest = ts.server.protocol.CompletionsRequest;
import CompletionEntry = ts.CompletionEntry;
import sys = ts.sys;
import {CompileInfoHolder} from "./compile-info-holder";


/**
 * Default tsserver implementation doesn't return response in most cases ("open", "close", etc.)
 * we want to override the behaviour and send empty-response holder
 */
const doneRequest:Response = {
    responseRequired: true,
    response: "done"
}

type Response = {response?:any; responseRequired?:boolean};
type ProjectResult = {succeeded:boolean; projectOptions?:ts.server.ProjectOptions; error?:ts.server.ProjectOpenResult};
export function getSession(ts_impl, logger:ts.server.Logger):ts.server.Session {

    let TypeScriptSession:typeof ts.server.Session = ts_impl.server.Session;
    let TypeScriptProjectService:typeof ts.server.ProjectService = ts_impl.server.ProjectService;
    let TypeScriptCommandNames:typeof ts.server.CommandNames = ts_impl.server.CommandNames;
    TypeScriptCommandNames.IDEChangeFiles = "ideChangeFiles";
    TypeScriptCommandNames.IDECompile = "ideCompile";
    TypeScriptCommandNames.IDEGetErrors = "ideGetErr";
    TypeScriptCommandNames.IDEGetErrors = "ideGetErr";
    TypeScriptCommandNames.IDECompletions = "ideCompletions";

    // var 
    class IDESession extends TypeScriptSession {
        private _host:ts.server.ServerHost;

        private _mySeq:number;


        constructor(host:ts.server.ServerHost,
                    byteLength:(buf:string, encoding?:string) => number,
                    hrtime:(start?:number[]) => number[],
                    logger:ts.server.Logger) {
            super(host, byteLength, hrtime, logger);
            this._host = host;
            let handler = this.projectService.eventHandler;
            //reuse handler
            this.projectService = new IDEProjectService(host, logger, handler);
        }

        public send(msg:ts.server.protocol.Message) {
            const json = JSON.stringify(msg);
            this._host.write(json + "\n");
        }

        executeCommand(request:ts.server.protocol.Request):Response {
            if (TypeScriptCommandNames.Open == request.command) {
                //use own implementation
                const openArgs = <ts.server.protocol.OpenRequestArgs>request.arguments;
                this.openClientFileExt(openArgs);
                return doneRequest;

            } else if (TypeScriptCommandNames.ReloadProjects == request.command) {
                this.getIDEProjectService().projectEmittedWithAllFiles = {};
                return super.executeCommand(request);
            } else if (TypeScriptCommandNames.IDEChangeFiles == request.command) {
                const updateFilesArgs = <ts.server.protocol.IDEUpdateFilesContentArgs>request.arguments;
                return this.updateFilesExt(updateFilesArgs);
            } else if (TypeScriptCommandNames.IDECompile == request.command) {
                const fileArgs = <ts.server.protocol.IDECompileFileRequestArgs>request.arguments;
                return this.compileFileExt(fileArgs);
            } else if (TypeScriptCommandNames.Close == request.command) {
                super.executeCommand(request);
                return doneRequest;
            } else if (TypeScriptCommandNames.IDEGetErrors == request.command) {
                let args = request.arguments;
                return {response: {infos: this.getDiagnosticsExt(args.files)}, responseRequired: true}
            } else if (TypeScriptCommandNames.IDECompletions == request.command) {
                let result:Response = super.executeCommand({
                    command: TypeScriptCommandNames.Completions,
                    arguments: request.arguments,
                    seq: request.seq,
                    type: request.type
                });
                const args = <ts.server.protocol.CompletionsRequestArgs>request.arguments;
                let response:ts.server.protocol.CompletionEntry[] = result.response;

                return {
                    response: this.getIDECompletions(args, response),
                    responseRequired: true
                }
            }

            return super.executeCommand(request);
        }

        updateFilesExt(args:ts.server.protocol.IDEUpdateFilesContentArgs):Response {
            let updated = false;
            let files = args.files;
            for (let fileName in files) {
                if (files.hasOwnProperty(fileName)) {
                    let content = files[fileName];
                    if (content) {
                        this.changeFileExt(fileName, content);
                        updated = true;
                    }
                }
            }

            if (args.filesToReloadContentFromDisk) {
                for (let fileName of args.filesToReloadContentFromDisk) {
                    if (!fileName) {
                        continue;
                    }

                    let file = ts_impl.normalizePath(fileName);
                    this.projectService.closeClientFile(file);


                    serverLogger("Reload file from disk " + file);
                    updated = true;
                }
            }

            if (updated) {
                this.updateProjectStructureExt();
                // this.projectService.updateProjectStructure();
            }

            return doneRequest;
        }

        private updateProjectStructureExt() {
            let mySeq = this.getChangeSeq();
            let matchSeq = (n) => n === mySeq;
            setTimeout(() => {
                if (matchSeq(this.getChangeSeq())) {
                    this.projectService.updateProjectStructure();
                }
            }, 1500);
        }

        private getChangeSeq() {
            let anyThis:any = this;
            let superClassSeq = anyThis.changeSeq;
            if (typeof superClassSeq !== "undefined") {
                return superClassSeq;
            }

            return this._mySeq;
        }

        openClientFileExt(openArgs:ts.server.protocol.OpenRequestArgs):{response?:any; responseRequired?:boolean} {
            const fileName = openArgs.file;
            const fileContent = openArgs.fileContent;
            const configFile = openArgs.tsConfig;
            const file = ts_impl.normalizePath(fileName);

            return (<IDEProjectService>this.projectService).openClientFileExt(file, fileContent, configFile);
        }

        changeFileExt(fileName:string, content:string, tsconfig?:string) {
            const file = ts_impl.normalizePath(fileName);
            const project = this.projectService.getProjectForFile(file);
            if (project) {
                const compilerService = project.compilerService;
                let scriptInfo = compilerService.host.getScriptInfo(file);
                if (scriptInfo != null) {
                    scriptInfo.svc.reload(content);
                    serverLogger("Reload content from text " + file);
                } else {
                    serverLogger("ScriptInfo is null " + file);
                }
            } else {
                serverLogger("Cannot find project for " + file);
                this.openClientFileExt({
                    file: fileName,
                    fileContent: content,
                    tsConfig: tsconfig
                });
            }
        }

        compileFileExt(req:ts.server.protocol.IDECompileFileRequestArgs):Response {
            let startCompile = this.getTime();

            let compileExactFile:boolean = req.file != null;
            if (!compileExactFile && !req.tsConfig) {
                return doneRequest;
            }

            let requestedFile:string = ts_impl.normalizePath(req.file ? req.file : req.tsConfig);
            let project:ts.server.Project = null;
            if (req.file) {
                project = this.projectService.getProjectForFile(requestedFile);
            } else {
                this.projectService.openOrUpdateConfiguredProjectForFile(requestedFile);
                project = this.projectService.findConfiguredProjectByConfigFile(requestedFile);
            }

            serverLogger("Get project end time: " + (this.getTime() - startCompile));
            const outFiles = [];
            let diagnostics:ts.server.protocol.DiagnosticEventBody[] = req.includeErrors ? [] : undefined;
            if (project) {
                let projectFilename = project.projectFilename;
                let languageService = project.compilerService.languageService;
                let program = languageService.getProgram();
                if (isLogEnabled) {
                    serverLogger("Get source files end time " + program.getSourceFiles().length + "(count): " + (this.getTime() - startCompile) + "ms");
                }
                let compileInfoHolder:CompileInfoHolder = this.getIDEProjectService().projectEmittedWithAllFiles[projectFilename];
                compileExactFile = compileExactFile && compileInfoHolder == null;
                if (!compileInfoHolder) {
                    compileInfoHolder = new CompileInfoHolder(ts_impl);
                    this.getIDEProjectService().projectEmittedWithAllFiles[projectFilename] = compileInfoHolder;
                }

                if (projectFilename && !compileExactFile) {
                    let toUpdateFiles:ts.SourceFile[] = [];
                    let rawSourceFiles = program.getSourceFiles();
                    rawSourceFiles.forEach((val) => {
                        if (compileInfoHolder.checkUpdateAndAddToCache(val)) {
                            toUpdateFiles.push(val);
                        }
                    });

                    let fileWriteCallback = this.getFileWrite(project, outFiles);
                    let compilerOptions = program.getCompilerOptions();
                    let useOutFile = compilerOptions && (compilerOptions.outFile || compilerOptions.out);
                    if (toUpdateFiles.length > 0) {
                        if (toUpdateFiles.length == rawSourceFiles.length || useOutFile) {
                            let emitResult:ts.EmitResult = program.emit(undefined, fileWriteCallback);
                            diagnostics = this.appendEmitDiagnostics(project, emitResult, diagnostics);
                        } else {
                            toUpdateFiles.forEach((el) => {
                                let emitResult:ts.EmitResult = program.emit(el, fileWriteCallback);
                                diagnostics = this.appendEmitDiagnostics(project, emitResult, diagnostics);
                            })
                        }
                    }


                    serverLogger("End emit files: " + (this.getTime() - startCompile));
                } else {
                    let sourceFile:ts.SourceFile = program.getSourceFile(requestedFile);
                    if (sourceFile) {
                        if (compileInfoHolder.checkUpdateAndAddToCache(sourceFile)) {
                            let emitResult:ts.EmitResult = project.program.emit(sourceFile, this.getFileWrite(project, outFiles));
                            diagnostics = this.appendEmitDiagnostics(project, emitResult, diagnostics);

                        }
                    } else {
                        serverLogger("Can't find source file: shouldn't be happened")
                    }
                }
            } else {
                serverLogger("Can't find project: shouldn't be happened")
            }

            if (diagnostics !== undefined) {
                diagnostics = diagnostics.concat(compileExactFile ?
                    this.getDiagnosticsExt([requestedFile], project) :
                    this.getProjectDiagnosticsExt(project)
                );
            }
            serverLogger("End get diagnostics stage: " + (this.getTime() - startCompile));

            return {response: {generatedFiles: outFiles, infos: diagnostics}, responseRequired: true};
        }

        private getIDEProjectService() {
            return (<IDEProjectService>this.projectService);
        }

        private getTime() {
            return new Date().getTime();
        }


        appendEmitDiagnostics(project:ts.server.Project, emitResult:ts.EmitResult, diagnostics:ts.server.protocol.DiagnosticEventBody[]) {
            if (diagnostics !== undefined && emitResult && emitResult.diagnostics) {
                let emitDiagnostics = emitResult.diagnostics;
                return diagnostics.concat(emitDiagnostics.map((el) => {
                        return {file: el.file.fileName, diagnostics: [formatDiagnostic(el.file.fileName, project, el)]}
                    }
                ));
            }

            return diagnostics;
        }

        getFileWrite(project, outFiles:string[]) {
            return (fileName, data?, writeByteOrderMark?, onError?, sourceFiles?) => {
                let normalizedName = ts_impl.normalizePath(fileName);
                this.ensureDirectoriesExist(ts_impl.getDirectoryPath(normalizedName));
                (<any>this._host).writeFile(normalizedName, data, writeByteOrderMark, onError, sourceFiles);
                outFiles.push(normalizedName);
            }
        }

        logError(err:Error, cmd:string) {
            const typedErr = <any>err;
            serverLogger("Error processing message: " + err.message + " " + typedErr.stack);

            super.logError(err, cmd);
        }

        getIDECompletions(req:ts.server.protocol.CompletionsRequestArgs,
                          entries:ts.server.protocol.CompletionEntry[]) {
            if (!entries) {
                return entries;
            }
            const file = ts_impl.normalizePath(req.file);
            const project = this.projectService.getProjectForFile(file);
            if (!project) {
                serverLogger("Can't find project: shouldn't be happened")
                return entries;
            }
            const compilerService = project.compilerService;
            const position = compilerService.host.lineOffsetToPosition(file, req.line, req.offset);
            let count = 0;
            return entries.reduce((accum:ts.server.protocol.CompletionEntry[], entry:CompletionEntry) => {
                if (count++ > 20) {
                    accum.push(entry);
                } else {
                    const details:any = compilerService.languageService.getCompletionEntryDetails(file, position, entry.name);
                    if (details) {
                        details.sortText = entry.sortText;
                        accum.push(details);
                    }
                }
                return accum;
            }, []);
        }

        /**
         * Possible we can remove the implementation if we will use 'pull' events
         * now just for test we use 'blocking' implementation
         * to check speed of processing
         * todo use 'pull' implementation
         */
        private getDiagnosticsExt(fileNames:string[], commonProject?:ts.server.Project):ts.server.protocol.DiagnosticEventBody[] {
            const checkList = fileNames.reduce((accumulator:ts.server.PendingErrorCheck[], fileName:string) => {
                fileName = ts_impl.normalizePath(fileName);
                if (commonProject) {
                    accumulator.push({fileName, project: commonProject});
                } else {
                    const project:ts.server.Project = this.projectService.getProjectForFile(fileName);
                    if (project) {
                        accumulator.push({fileName, project});
                    }
                }
                return accumulator;
            }, []);

            let result:ts.server.protocol.DiagnosticEventBody[] = [];

            if (checkList.length > 0) {
                for (let checkSpec of checkList) {
                    let file:string = checkSpec.fileName;
                    let project = checkSpec.project;

                    if (project.getSourceFileFromName(file, true)) {
                        let diagnostics:ts.server.protocol.Diagnostic[] = [];
                        const syntacticDiagnostics:ts.Diagnostic[] = project.compilerService.languageService.getSyntacticDiagnostics(file);
                        if (syntacticDiagnostics) {
                            const bakedDiagnostics = syntacticDiagnostics.map((el:ts.Diagnostic) => formatDiagnostic(file, checkSpec.project, el));
                            diagnostics = diagnostics.concat(bakedDiagnostics);
                        }

                        const semanticDiagnostics = project.compilerService.languageService.getSemanticDiagnostics(file);
                        if (semanticDiagnostics) {
                            const bakedSemanticDiagnostics = semanticDiagnostics.map((el:ts.Diagnostic) => formatDiagnostic(file, checkSpec.project, el));
                            diagnostics = diagnostics.concat(bakedSemanticDiagnostics);

                        }

                        result.push({
                            file,
                            diagnostics
                        })
                    }
                }
            }

            return result;
        }

        getProjectDiagnosticsExt(project:ts.server.Project):ts.server.protocol.DiagnosticEventBody[] {
            if (!project) {
                return []
            }
            return this.getDiagnosticsExt(project.getFileNames(), project);
        }

        ensureDirectoriesExist(directoryPath:string) {
            if (directoryPath.length > ts_impl.getRootLength(directoryPath) && !this._host.directoryExists(directoryPath)) {
                const parentDirectory = ts_impl.getDirectoryPath(directoryPath);
                this.ensureDirectoriesExist(parentDirectory);
                this._host.createDirectory(directoryPath);
            }
        }
    }

    class IDEProjectService extends TypeScriptProjectService {

        projectEmittedWithAllFiles:{
            [p:string]:CompileInfoHolder
        } = {}

        constructor(host:ts.server.ServerHost,
                    psLogger:ts.server.Logger,
                    eventHandler:ts.server.ProjectServiceEventHandler) {
            super(host, psLogger, eventHandler);
        }

        openClientFileExt(fileName:string, fileContent:string, configFileName:string) {
            if (configFileName) {
                serverLogger("Open for specified tsconfig");
                this.openOrUpdateConfiguredProjectForFile(ts_impl.normalizePath(configFileName));
            } else {
                serverLogger("Try to find tsconfig");
                this.openOrUpdateConfiguredProjectForFile(fileName);
            }
            const info = this.openFile(fileName, /*openedByClient*/ true, fileContent);
            this.addOpenFile(info);
            return info;
        }


        watchedProjectConfigFileChanged(project:ts.server.Project):void {
            let projectFilename = project.projectFilename;
            super.watchedProjectConfigFileChanged(project);
            if (projectFilename) {
                this.projectEmittedWithAllFiles[projectFilename] = null;
            }
        }

        configFileToProjectOptions(configFilename:string):ProjectResult {
            function getBaseFileName(path) {
                if (path === undefined) {
                    return undefined;
                }
                let i = path.lastIndexOf(ts_impl.directorySeparator);
                return i < 0 ? path : path.substring(i + 1);
            }

            let configFileToProjectOptions:ProjectResult = super.configFileToProjectOptions(configFilename);

            if (configFileToProjectOptions && configFileToProjectOptions.projectOptions) {
                let projectOptions = configFileToProjectOptions.projectOptions;
                let files = projectOptions.files;

                if (files) {
                    let compilerOptions = projectOptions.compilerOptions;
                    let extensions:string[] = ts_impl.getSupportedExtensions(compilerOptions);
                    let newFiles = [];

                    l: for (let file of files) {
                        let fileName = getBaseFileName(file);
                        for (let extension of extensions) {
                            if (fileName.lastIndexOf(extension) > 0) {
                                newFiles.push(file);
                                continue l;
                            }
                        }
                        for (let extension of extensions) {
                            if (this.host.fileExists(file + extension)) {
                                newFiles.push(file + extension);
                                continue l;
                            }
                        }

                        newFiles.push(file);
                    }

                    let newOptions:ProjectResult = {
                        succeeded: configFileToProjectOptions.succeeded,
                        projectOptions: {
                            compilerOptions: compilerOptions,
                            files: newFiles
                        }
                    }
                    if (configFileToProjectOptions.error) {
                        newOptions.error = configFileToProjectOptions.error;
                    }

                    return newOptions;
                }
            }

            return configFileToProjectOptions;
        }
    }

    /**
     * copy formatDiag method (but we use 'TS' prefix)
     */
    function formatDiagnostic(fileName:string, project:ts.server.Project, diagnostic:ts.Diagnostic):ts.server.protocol.Diagnostic {
        return {
            start: project.compilerService.host.positionToLineOffset(fileName, diagnostic.start),
            end: project.compilerService.host.positionToLineOffset(fileName, diagnostic.start + diagnostic.length),
            text: "TS" + diagnostic.code + ":" + ts_impl.flattenDiagnosticMessageText(diagnostic.messageText, "\n")
        };
    }


    return new IDESession(ts_impl.sys, Buffer.byteLength, process.hrtime, logger);
}






