/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.contexts.model.graph;

import com.intellij.openapi.util.Pair;
import com.intellij.util.Function;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.graph.Graph;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Map;

/**
 * @since 14.1
 */
public abstract class LazyDependenciesGraph<N, E> implements Graph<N> {

  private final Map<N, Collection<Pair<N, E>>> myIns = ContainerUtil.newConcurrentMap();
  private final Map<N, Collection<Pair<N, E>>> myOuts = ContainerUtil.newConcurrentMap();

  private volatile boolean myIsBuilt = false;

  private final Function<Pair<N, E>, N> mySourceNodeFunction = pair -> pair.getFirst();

  @NotNull
  protected abstract Collection<Pair<N, E>> getDependencies(@NotNull N n);

  @Override
  public Iterator<N> getIn(N n) {
    guaranteeGraphIsBuilt();
    return getIterator(myIns.get(n));
  }

  @Override
  public Iterator<N> getOut(N n) {
    return getIterator(getOrCreateOutDependencies(n));
  }

  @NotNull
  private Iterator<N> getIterator(Collection<Pair<N, E>> dependencies) {
    return ContainerUtil.map(dependencies, mySourceNodeFunction).iterator();
  }

  protected final void guaranteeGraphIsBuilt() {
    if (!myIsBuilt) {
      final Collection<N> nodes = getNodes();
      for (N node : nodes) {
        getOrCreateOutDependencies(node);
      }
      myIsBuilt = true;
    }
  }

  @NotNull
  public final Collection<Pair<N, E>> getOrCreateOutDependencies(@NotNull N n) {
    Collection<Pair<N, E>> outSet = myOuts.get(n);
    if (outSet == null) {
      outSet = getDependencies(n);
      for (Pair<N, E> dependency : outSet) {
        addInDependency(n, dependency.first, dependency.second);
      }
      myOuts.put(n, outSet);
    }
    return myOuts.get(n);
  }

  public void addInDependency(@NotNull N from, @NotNull N to, @NotNull E edgeDescriptor) {
    Collection<Pair<N, E>> inNodes = myIns.get(to);
    if (inNodes == null) {
      inNodes = new LinkedHashSet<Pair<N, E>>();
    }
    inNodes.add(Pair.create(from, edgeDescriptor));
    myIns.put(to, inNodes);
  }
}
