/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.converters;

import com.intellij.jam.JamSimpleReferenceConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.lang.properties.PropertiesImplUtil;
import com.intellij.lang.properties.psi.PropertiesFile;
import com.intellij.lang.properties.psi.impl.PropertiesFileImpl;
import com.intellij.openapi.util.Condition;
import com.intellij.psi.*;
import com.intellij.spring.model.utils.SpringReferenceUtils;
import com.intellij.spring.model.utils.resources.SpringResourcesBuilder;
import com.intellij.spring.model.utils.resources.SpringResourcesUtil;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public class PropertiesFileConverter extends JamSimpleReferenceConverter<Set<PropertiesFile>> {

  @Override
  public Set<PropertiesFile> fromString(@Nullable String s, JamStringAttributeElement<Set<PropertiesFile>> context) {
    final PsiAnnotationMemberValue psiElement = context.getPsiElement();
    if (s != null && psiElement != null) {
      return new LinkedHashSet<>(getResourceFiles(psiElement, s, ",",
                                                  item -> item instanceof PropertiesFileImpl));
    }
    return Collections.emptySet();
  }

  @NotNull
  @Override
  public PsiReference[] createReferences(JamStringAttributeElement<Set<PropertiesFile>> context) {
    return getReferences(context);
  }

  private static PsiReference[] getReferences(JamStringAttributeElement<Set<PropertiesFile>> context) {
    PsiLiteral psiLiteral = context.getPsiLiteral();
    if (psiLiteral == null) return PsiReference.EMPTY_ARRAY;

    String s = ElementManipulators.getValueText(psiLiteral);

    return SpringResourcesUtil.getInstance()
      .getReferences(SpringResourcesBuilder.create(psiLiteral, s).fromRoot(s.startsWith("/")).soft(false));
  }

  private static List<PropertiesFile> getResourceFiles(final @NotNull PsiElement element,
                                                       final @NotNull String s,
                                                       final String delimiter,
                                                       final Condition<PsiFileSystemItem> filter) {
    final List<PsiReference> references = new ArrayList<>();
    final int startInElement = ElementManipulators.getOffsetInElement(element);

    SpringReferenceUtils.processSeparatedString(s, delimiter, (separatedString, offset) -> {
      final SpringResourcesBuilder builder =
        SpringResourcesBuilder.create(element, separatedString).offset(offset.intValue() + startInElement);
      ContainerUtil.addAll(references, SpringResourcesUtil.getInstance().getReferences(builder));
      return true;
    });

    Collection<PsiFile> files = SpringResourcesUtil.getInstance().getResourceItems(
      references.toArray(new PsiReference[references.size()]), filter);

    return ContainerUtil.mapNotNull(files, PropertiesImplUtil::getPropertiesFile);
  }
}
