/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.utils.filters;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiPackage;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.PackageScope;
import com.intellij.psi.search.searches.AnnotatedElementsSearch;
import com.intellij.spring.model.jam.stereotype.CustomSpringComponent;
import com.intellij.spring.model.jam.stereotype.SpringStereotypeElement;
import com.intellij.util.Processor;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.hash.HashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.Set;

public class SpringContextIncludeAnnotationFilter extends SpringContextFilter.IncludeExpression {

  public SpringContextIncludeAnnotationFilter(@Nullable String expression) {
    super(expression);
  }

  @NotNull
  @Override
  public Set<SpringStereotypeElement> includeStereotypes(@NotNull Module module, @NotNull Set<PsiPackage> packages) {
    final String annotation = getExpression();
    if (!StringUtil.isEmptyOrSpaces(annotation)) {
      GlobalSearchScope searchScope = GlobalSearchScope.moduleWithDependenciesAndLibrariesScope(module);
      final PsiClass annotationClass = JavaPsiFacade.getInstance(module.getProject()).findClass(annotation, searchScope);

      return getAnnotatedStereotypes(annotationClass, searchScope, packages);
    }
    return Collections.emptySet();
  }

  @NotNull
  public static Set<SpringStereotypeElement> getAnnotatedStereotypes(@Nullable PsiClass annotationClass,
                                                                     @NotNull GlobalSearchScope searchScope,
                                                                     @NotNull Set<PsiPackage> packages) {
    final Set<SpringStereotypeElement> components = new HashSet<SpringStereotypeElement>();
    if (annotationClass != null) {
      final Set<PsiClass> annotatedClasses = getAnnotatedClasses(annotationClass, packages, searchScope);
      for (PsiClass annotatedClass : annotatedClasses) {
        components.add(new CustomSpringComponent(annotationClass.getQualifiedName(), annotatedClass));
      }
    }

    return components;
  }

  public static Set<PsiClass> getAnnotatedClasses(@NotNull PsiClass annotationClass,
                                                  @NotNull Set<PsiPackage> packages,
                                                  @NotNull GlobalSearchScope searchScope) {
    final Set<PsiClass> annotatedClasses = new HashSet<PsiClass>();
    Processor<PsiClass> processor = psiClass -> {
      ContainerUtil.addIfNotNull(psiClass, annotatedClasses);
      return true;
    };

    if (annotationClass.isAnnotationType()) {
      for (PsiPackage aPackage : packages) {
        final GlobalSearchScope scope = searchScope.intersectWith(PackageScope.packageScope(aPackage, true));
        AnnotatedElementsSearch.searchPsiClasses(annotationClass, scope).forEach(processor);
      }
    }
    return annotatedClasses;
  }
}
