/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database.psi;

import com.intellij.database.dialects.DatabaseDialect;
import com.intellij.database.model.*;
import com.intellij.database.util.DasUtil;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.util.containers.ContainerUtil;
import icons.DatabaseIcons;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.Map;
import java.util.Set;

/**
 * @author gregsh
 */
public class DbPresentation {

  private static final Map<ObjectKind, Icon> ourIcons = ContainerUtil.newConcurrentMap();

  static {
    //ourIcons.put(ObjectKind.NONE, null);
    ourIcons.put(ObjectKind.DATABASE, DatabaseIcons.Database);
    ourIcons.put(ObjectKind.SCHEMA, DatabaseIcons.Schema);
    ourIcons.put(ObjectKind.SEQUENCE, DatabaseIcons.Sequence);
    ourIcons.put(ObjectKind.TABLE, DatabaseIcons.Table);
    ourIcons.put(ObjectKind.COLUMN, DatabaseIcons.Col);
    ourIcons.put(ObjectKind.INDEX, DatabaseIcons.Index);
    ourIcons.put(ObjectKind.KEY, DatabaseIcons.GoldKeyAlt);
    ourIcons.put(ObjectKind.FOREIGN_KEY, DatabaseIcons.BlueKey);
    //ourIcons.put(ObjectKind.RULE, DatabaseIcons.Rule);
    //ourIcons.put(ObjectKind.CLUSTER, null);
    ourIcons.put(ObjectKind.VIEW, DatabaseIcons.View);
    ourIcons.put(ObjectKind.MAT_VIEW, DatabaseIcons.MaterializedView);
    ourIcons.put(ObjectKind.CHECK, DatabaseIcons.CheckConstraint);
    ourIcons.put(ObjectKind.TRIGGER, DatabaseIcons.Trigger);
    ourIcons.put(ObjectKind.PACKAGE, DatabaseIcons.Package);
    ourIcons.put(ObjectKind.ALIAS_TYPE, DatabaseIcons.ObjecType);  //AliasType
    ourIcons.put(ObjectKind.TABLE_TYPE, DatabaseIcons.ObjecType);  //TableType
    ourIcons.put(ObjectKind.OBJECT_TYPE, DatabaseIcons.ObjecType);
    ourIcons.put(ObjectKind.OBJECT_ATTRIBUTE, DatabaseIcons.ObjecTypeAttribute);
    ourIcons.put(ObjectKind.COLLECTION_TYPE, DatabaseIcons.CollectionType);
    ourIcons.put(ObjectKind.BODY, DatabaseIcons.Body);
    ourIcons.put(ObjectKind.ROUTINE, DatabaseIcons.Routine);
    ourIcons.put(ObjectKind.ARGUMENT, DatabaseIcons.Argument);
    ourIcons.put(ObjectKind.OPERATOR, DatabaseIcons.Operator);
    ourIcons.put(ObjectKind.VARIABLE, DatabaseIcons.Variable);
    ourIcons.put(ObjectKind.SYNONYM, DatabaseIcons.Synonym);
    //ourIcons.put(ObjectKind.DB_LINK, null);
    //ourIcons.put(ObjectKind.COMMENT, null);
    //ourIcons.put(ObjectKind.DOMESTIC_COLUMN_REF, null);
    //ourIcons.put(ObjectKind.UNKNOWN_OBJECT, null);
  }

  @Nullable
  public static Icon getIcon(ObjectKind kind) {
    return ourIcons.get(kind);
  }

  public static void registerIcon(ObjectKind kind, Icon icon) {
    ourIcons.put(kind, icon);
  }

  @Nullable
  public static Icon getIcon(@NotNull DasObject c) {
    ObjectKind kind = c.getKind();
    if (c instanceof PsiElement && !((PsiElement)c).isValid()) {
      return getIcon(kind);
    }
    if (kind == ObjectKind.COLUMN && c instanceof DasColumn) {
      return getColumnIcon((DasColumn)c);
    }
    else if (kind == ObjectKind.KEY && c instanceof DasTableKey) {
      return ((DasTableKey)c).isPrimary() ? DatabaseIcons.GoldKey : DatabaseIcons.GoldKeyAlt;
    }
    else if (kind == ObjectKind.INDEX && c instanceof DasIndex) {
      return ((DasIndex)c).isUnique() ? DatabaseIcons.IndexUnique : DatabaseIcons.Index;
    }
    else if (c instanceof DasRoutine) {
      DasRoutine.Kind k = ((DasRoutine)c).getRoutineKind();
      return k == DasRoutine.Kind.PROCEDURE ? DatabaseIcons.Procedure :
             k == DasRoutine.Kind.FUNCTION ? DatabaseIcons.Function :
             DatabaseIcons.Routine;
    }
    return getIcon(kind);
  }

  static Icon getColumnIcon(@NotNull DasColumn c) {
    Set<DasColumn.Attribute> attrs = c.getTable().getColumnAttrs(c);
    boolean pinned = c.isNotNull();
    boolean inIndex = attrs.contains(DasColumn.Attribute.INDEX);
    boolean pk = attrs.contains(DasColumn.Attribute.PRIMARY_KEY);
    boolean fk = attrs.contains(DasColumn.Attribute.FOREIGN_KEY);
    if (pk && fk) {
      if (pinned) {
        return inIndex ? DatabaseIcons.ColGoldBlueKeyDotIndex : DatabaseIcons.ColGoldBlueKeyDot;
      }
      else {
        return inIndex ? DatabaseIcons.ColGoldBlueKeyIndex : DatabaseIcons.ColGoldBlueKey;
      }
    }
    else if (pk) {
      if (pinned) {
        return inIndex ? DatabaseIcons.ColGoldKeyDotIndex : DatabaseIcons.ColGoldKeyDot;
      }
      else {
        return inIndex ? DatabaseIcons.ColGoldKeyIndex : DatabaseIcons.ColGoldKey;
      }
    }
    else if (fk) {
      if (pinned) {
        return inIndex ? DatabaseIcons.ColBlueKeyDotIndex : DatabaseIcons.ColBlueKeyDot;
      }
      else {
        return inIndex ? DatabaseIcons.ColBlueKeyIndex : DatabaseIcons.ColBlueKey;
      }
    }
    else {
      if (pinned) {
        return inIndex ? DatabaseIcons.ColDotIndex : DatabaseIcons.ColDot;
      }
      else {
        return inIndex ? DatabaseIcons.ColIndex : DatabaseIcons.Col;
      }
    }
  }

  @NotNull
  public static String getPresentableName(@NotNull DasObject o) {
    String name = o.getName();
    return name == DasUtil.NO_NAME ? "<unnamed>" : name;
  }

  @NotNull
  public static String getPresentableName(@NotNull ObjectKind o, @NotNull DatabaseDialect dialect) {
    String name = dialect.getNameForKind(o);
    return name == null ? DasUtil.NO_NAME : StringUtil.toLowerCase(name);
  }
}
