/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package com.intellij.javaee.model.jam;

import com.intellij.jam.JamElement;
import com.intellij.jam.JamSimpleReferenceConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.annotations.JamPsiConnector;
import com.intellij.jam.annotations.JamPsiValidity;
import com.intellij.jam.reflect.*;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.javaee.web.*;
import com.intellij.javaee.web.facet.WebFacet;
import com.intellij.openapi.util.Condition;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiElementRef;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.GenericValue;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * @author Dmitry Avdeev
 */
public abstract class JamFilter extends CommonModelElement.PsiBase implements CommonFilter, CommonServletMapping<CommonFilter>, JamElement {

  public static final JamStringAttributeMeta.Single<String> NAME_META = JamAttributeMeta.singleString("filterName");
  public static final JamStringAttributeMeta.Collection<String> URL_PATTERNS_META = JamAttributeMeta.collectionString("urlPatterns");
  public static final JamStringAttributeMeta.Collection<CommonServlet> SERVLET_NAMES_META = JamAttributeMeta.collectionString("servletNames", new JamSimpleReferenceConverter<CommonServlet>() {

    @Override
    public CommonServlet fromString(@Nullable final String s, JamStringAttributeElement<CommonServlet> context) {
      if (s == null) {
        return null;
      }
      WebFacet facet = WebUtil.getWebFacet(context.getPsiElement());
      if (facet == null) {
        return null;
      }
      return ContainerUtil.find(facet.getWebModel().getServlets(),
                                commonServlet -> s.equals(commonServlet.getServletName().getStringValue()));
    }

    @Override
    public Collection<CommonServlet> getVariants(JamStringAttributeElement<CommonServlet> context) {
      WebFacet facet = WebUtil.getWebFacet(context.getPsiElement());
      if (facet == null) {
        return Collections.emptyList();
      }
      return facet.getWebModel().getServlets();
    }
  });

  public static final JamAnnotationAttributeMeta.Collection<JamInitParam> INIT_PARAMS_META =
    JamAttributeMeta.annoCollection("initParams", JamInitParam.INIT_PARAM_ANNO_META, JamInitParam.class);

  @NonNls public static final String ANNO_NAME = WebCommonClassNames.ANNOTATION_WEB_FILTER;

  public static final JamAnnotationMeta FILTER_ANNO_META = new JamAnnotationMeta(ANNO_NAME).
    addAttribute(NAME_META).
    addAttribute(URL_PATTERNS_META).
    addAttribute(INIT_PARAMS_META).
    addAttribute(SERVLET_NAMES_META);

  public static final JamClassMeta<JamFilter> FILTER_CLASS_META = new JamClassMeta<JamFilter>(JamFilter.class).
    addAnnotation(FILTER_ANNO_META);

  @JamPsiConnector
  public abstract PsiClass getPsiClass();

  @JamPsiValidity
  public abstract boolean isPsiValid();

  @NotNull
  @Override
  public PsiElement getPsiElement() {
    return getPsiClass();
  }

  @NotNull
  public JamStringAttributeElement<String> getFilterName() {
    return FILTER_ANNO_META.getAttribute(getPsiClass(), NAME_META);
  }

  public List<JamStringAttributeElement<String>> getUrlPatterns() {
    return FILTER_ANNO_META.getAttribute(getPsiClass(), URL_PATTERNS_META);
  }

  public List<? extends GenericValue<CommonServlet>> getServletNames() {
    return SERVLET_NAMES_META.getJam(FILTER_ANNO_META.getAnnotationRef(getPsiClass()));
  }

  public List<JamInitParam> getInitParams() {
    return FILTER_ANNO_META.getAttribute(getPsiClass(), INIT_PARAMS_META);
  }

  public JamInitParam addInitParam() {
    return INIT_PARAMS_META.addAttribute(PsiElementRef.real(FILTER_ANNO_META.getAnnotation(getPsiClass())));
  }

  public CommonFilter getServlet() {
    return this;
  }

  public PsiElement getMappingElement() {
    return FILTER_ANNO_META.getAnnotation(getPsiClass());
  }
}
