/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.javaee.web;

import com.intellij.openapi.util.TextRange;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.psi.PsiReferenceBase;
import com.intellij.util.xml.GenericValue;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author Dmitry Avdeev
 */
public abstract class ServletMappingInfo {
  private final String myUrlPattern;
  private final CommonServletMapping<CommonServlet> myServletMapping;
  private final ServletMappingType myType;

  public static List<ServletMappingInfo> createMappingInfos(CommonServletMapping mapping) {
    List<GenericValue<String>> patterns = mapping.getUrlPatterns();
    if (patterns.isEmpty()) {
      return Collections.emptyList();
    }
    ArrayList<ServletMappingInfo> infos = new ArrayList<ServletMappingInfo>(patterns.size());
    for (GenericValue<String> pattern : patterns) {
      String s = pattern.getStringValue();
      if (s != null) {
        ServletMappingInfo mappingInfo = ServletMappingType.getPatternType(s).createMappingInfo(s, mapping);
        infos.add(mappingInfo);
      }
    }
    return infos;
  }

  protected ServletMappingInfo(@NotNull String urlPattern, CommonServletMapping<CommonServlet> mapping, ServletMappingType type) {
    myUrlPattern = urlPattern;
    myServletMapping = mapping;
    myType = type;
  }

  public boolean equals(final Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    final ServletMappingInfo info = (ServletMappingInfo)o;

    if (!myServletMapping.equals(info.myServletMapping)) return false;
    if (myType != info.myType) return false;
    if (!myUrlPattern.equals(info.myUrlPattern)) return false;

    return true;
  }

  public int hashCode() {
    int result;
    result = myUrlPattern.hashCode();
    result = 31 * result + myServletMapping.hashCode();
    result = 31 * result + myType.hashCode();
    return result;
  }

  @NotNull
  public final CommonServletMapping getServletMapping() {
    return myServletMapping;
  }

  public CommonServlet getServlet() {
    //noinspection ConstantConditions
    return myServletMapping.getServlet();
  }

  @NotNull
  public final String getUrlPattern() {
    return myUrlPattern;
  }

  @NotNull
  public final ServletMappingType getType() {
    return myType;
  }

  @Nullable
  public abstract TextRange getNameRange(String url);

  @Nullable
  public abstract TextRange getMappingRange(String url);

  public abstract String stripMapping(@NotNull String url);

  public abstract String addMapping(@NotNull String url);

  public boolean matches(@NotNull String url) {
    return getMappingRange(url) != null;
  }

  @Nullable
  public PsiReference getMappingReference(PsiElement element, String actionUrl, final boolean soft) {
    TextRange range = getMappingRange(actionUrl);
    if (range == null) {
      return null;
    }
    PsiReferenceBase<PsiElement> ref = new PsiReferenceBase<PsiElement>(element, range, soft) {

      @Nullable
      public PsiElement resolve() {
        assert myServletMapping != null;
        return myServletMapping.getMappingElement();
      }

      @NotNull
      public Object[] getVariants() {
        return EMPTY_ARRAY;
      }
    };
    final TextRange textRange = range.shiftRight(ref.getRangeInElement().getStartOffset());
    ref.setRangeInElement(textRange);
    return ref;
  }

  public static class ExtensionMappingInfo extends ServletMappingInfo {
    private final String mySubstring;

    public ExtensionMappingInfo(final String urlPattern, CommonServletMapping mapping) {
      super(urlPattern, mapping, ServletMappingType.EXTENSION);
      mySubstring = urlPattern.substring(1);
    }

    public boolean equals(final Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      if (!super.equals(o)) return false;

      final ExtensionMappingInfo that = (ExtensionMappingInfo)o;

      if (!mySubstring.equals(that.mySubstring)) return false;

      return true;
    }

    public int hashCode() {
      int result = super.hashCode();
      result = 31 * result + mySubstring.hashCode();
      return result;
    }

    @Nullable
    public TextRange getNameRange(String url) {
      int pos = url.indexOf(mySubstring);
      if (pos != -1 && pos == url.length() - mySubstring.length()) {
        return new TextRange(0, pos);
      }
      return null;
    }

    @Nullable
    public TextRange getMappingRange(String actionUrl) {
      int pos = actionUrl.indexOf(mySubstring);
      if (pos != -1 && pos == actionUrl.length() - mySubstring.length()) {
        return new TextRange(pos + 1, actionUrl.length());
      }
      return null;
    }

    @Nullable
    public String stripMapping(@NotNull String actionUrl) {
      int pos = actionUrl.indexOf(mySubstring);
      if (pos != -1 && pos == actionUrl.length() - mySubstring.length()) {
        return actionUrl.substring(0, pos);
      }
      return actionUrl;
    }

    public String addMapping(@NotNull String url) {
      return url + mySubstring;
    }
  }

  public static class PathMappingInfo extends ServletMappingInfo {
    private final String myMappingSubstring;
    public PathMappingInfo(final String urlPattern, CommonServletMapping mapping) {
      super(urlPattern, mapping, ServletMappingType.PATH);
      myMappingSubstring = urlPattern.substring(0, urlPattern.length() - 2);
    }

    public boolean equals(final Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      if (!super.equals(o)) return false;

      final PathMappingInfo that = (PathMappingInfo)o;

      if (!myMappingSubstring.equals(that.myMappingSubstring)) return false;

      return true;
    }

    public int hashCode() {
      int result = super.hashCode();
      result = 31 * result + myMappingSubstring.hashCode();
      return result;
    }

    @Nullable
    public TextRange getNameRange(String url) {
      int pos = url.indexOf(myMappingSubstring.isEmpty() ? "/" : myMappingSubstring);
      if (pos == 0 && url.length() > myMappingSubstring.length()) {
        return new TextRange(myMappingSubstring.length() + 1, url.length());
      }
      return null;
    }

    @Nullable
    public TextRange getMappingRange(String url) {
      int pos = url.indexOf(myMappingSubstring);
      if (pos == 0) {
        return new TextRange(0, myMappingSubstring.length());
      }
      return null;
    }

    @Nullable
    public String stripMapping(@NotNull String url) {
      int pos = url.indexOf(myMappingSubstring);
      if (pos == 0) {
        return url.substring(myMappingSubstring.length());
      }
      return null;
    }

    public String addMapping(@NotNull String url) {
      return myMappingSubstring + url;
    }
  }

  public static class ExactMappingInfo extends ServletMappingInfo {

    public ExactMappingInfo(final String urlPattern, CommonServletMapping mapping) {
      super(urlPattern, mapping, ServletMappingType.EXACT);
    }

    @Nullable
    public TextRange getNameRange(String url) {
      return WebUtil.trimRange(url, new TextRange(0, url.length()));
    }

    @Nullable
    public TextRange getMappingRange(String url) {
      return null;
    }

    @Nullable
    public String stripMapping(@NotNull String url) {
      return url;
    }

    public String addMapping(@NotNull String url) {
      return url;
    }
  }

  public static class DefaultMappingInfo extends ServletMappingInfo {
    public DefaultMappingInfo(final String urlPattern, final CommonServletMapping mapping) {
      super(urlPattern, mapping, ServletMappingType.DEFAULT);
    }

    @Nullable
    public TextRange getNameRange(final String url) {
      return null;
    }

    @Nullable
    public TextRange getMappingRange(final String url) {
      return url.equals("/") ? TextRange.from(0, 1) : null;
    }

    public String stripMapping(@NotNull final String url) {
      return url;
    }

    public String addMapping(@NotNull final String url) {
      return url;
    }
  }
}
