package com.intellij.lang.css;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.extensions.Extensions;
import com.intellij.openapi.module.Module;
import com.intellij.psi.css.CssBundle;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public abstract class CssDialect {
  public static final ExtensionPointName<CssDialect> EP_NAME = ExtensionPointName.create("com.intellij.css.dialect");
  public static final CssDialect CLASSIC = new CssDialect() {
    @Override
    public String getName() {
      return "CLASSIC";
    }

    @Override
    public String getDisplayName() {
      return CssBundle.message("css.w3c.dialect.name");
    }

    @Override
    public boolean isDefault(@NotNull Module module) {
      return true;
    }
  };

  public abstract String getName();
  public abstract String getDisplayName();
  public abstract boolean isDefault(@NotNull Module module);

  @Nullable
  public static CssDialect getDefaultDialect(@Nullable Module module) {
    if (module != null) {
      for (CssDialect dialect : Extensions.getExtensions(EP_NAME)) {
        if (dialect.isDefault(module)) return dialect;
      }
      return CLASSIC;
    }
    return null;
  }

  public static List<CssDialect> getAvailableDialects() {
    final List<CssDialect> results = new ArrayList<CssDialect>();
    Collections.addAll(results, Extensions.getExtensions(EP_NAME));
    results.add(CLASSIC);
    return results;
  }

  public static List<CssDialect> getCustomDialects() {
    final List<CssDialect> results = new ArrayList<CssDialect>();
    Collections.addAll(results, Extensions.getExtensions(EP_NAME));
    return results;
  }
}
