/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database.dataSource;

import com.intellij.database.model.DasModel;
import com.intellij.database.model.DatabaseSystem;
import com.intellij.database.util.Casing;
import com.intellij.database.util.DasUtil;
import com.intellij.openapi.util.SimpleModificationTracker;
import com.intellij.openapi.util.text.StringUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Prefer com.intellij.database.model.DatabaseSystem.
 * @see DatabaseSystem
 * @see com.intellij.database.psi.DbDataSource
 *
 * @deprecated
 *
 * @author cdr
 */
public abstract class DataSource extends SimpleModificationTracker implements DatabaseSystem {


  public abstract boolean isGlobal();

  public abstract String getUniqueId();

  public abstract boolean isReadOnly();

  @NotNull
  public abstract String getName();

  @Nullable
  public abstract String getComment();

  public String getDatabaseProductName() {
    return getInfo().myProductName;
  }

  public String getDatabaseProductVersion() {
    return getInfo().myProductVersion;
  }

  public String getJDBCVersion() {
    return getInfo().myJDBCVersion;
  }

  public String getDriverName() {
    return getInfo().myDriverName;
  }

  public String getDriverVersion() {
    return getInfo().myDriverVersion;
  }

  public Casing getCaseModes() {
    return getInfo().myCasing;
  }

  @NotNull
  public abstract DasModel getModel();

  protected abstract Info getInfo();

  public abstract String getSourceName();

  /**
   * Should be used in queries and not in DatabaseMetaData inquiries
   * @param identifier database entity name
   * @return identifier or its quoted version
   */
  @Nullable
  public String quoteIdentifierIfNeeded(@Nullable final String identifier) {
    if (getInfo().myIdentifierQuoteString == null || identifier == null) return identifier;
    if (identifier.startsWith(getInfo().myIdentifierQuoteString)) return identifier;
    for (int i = 0, len = identifier.length(); i<len; i++) {
      final char c = identifier.charAt(i);
      if (i == 0 && !StringUtil.isJavaIdentifierStart(c) || i > 0 && !StringUtil.isJavaIdentifierPart(c)) {
        if (getInfo().myExtraNameCharacters == null || getInfo().myExtraNameCharacters.indexOf(c) == -1) {
          return quoteIdentifier(identifier);
        }
      }
    }
    return identifier;
  }

  @Nullable
  public String quoteIdentifier(@Nullable final String identifier) {
    if (getInfo().myIdentifierQuoteString == null || identifier == null) return identifier;
    return getInfo().myIdentifierQuoteString + identifier + getInfo().myIdentifierQuoteString;
  }

  @NotNull
  @Override
  public String toString() {
    return getClass().getSimpleName() + "@" + Integer.toHexString(hashCode()) + ": " + getName();
  }

  @Override
  public boolean equals(Object o) {
    return this == o;
  }

  @Override
  public int hashCode() {
    return super.hashCode();
  }

  static class Info {
    String myProductName;
    String myProductVersion;
    String myJDBCVersion;
    @Nullable String myIdentifierQuoteString;
    @Nullable String myExtraNameCharacters;
    String myDriverName;
    String myDriverVersion;
    Casing myCasing = DasUtil.CASING_MIXED;

    void copyFrom(@NotNull Info info) {
      myProductName = info.myProductName;
      myProductVersion = info.myProductVersion;
      myJDBCVersion = info.myJDBCVersion;
      myIdentifierQuoteString = info.myIdentifierQuoteString;
      myExtraNameCharacters = info.myExtraNameCharacters;
      myDriverName = info.myDriverName;
      myDriverVersion = info.myDriverVersion;
      myCasing = info.myCasing;
    }
  }
}
