/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.converters;

import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.codeInspection.LocalQuickFix;
import com.intellij.codeInspection.LocalQuickFixProvider;
import com.intellij.openapi.util.TextRange;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiClassType;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.spring.SpringApiBundle;
import com.intellij.spring.contexts.model.SpringModel;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.converters.fixes.bean.SpringBeanResolveQuickFixManager;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.spring.model.utils.SpringModelSearchers;
import com.intellij.spring.model.xml.DomSpringBean;
import com.intellij.spring.model.xml.beans.Beans;
import com.intellij.util.ReflectionUtil;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.ConvertContext;
import com.intellij.util.xml.GenericDomValue;
import com.intellij.util.xml.converters.DelimitedListConverter;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * @author Yann C&eacute;bron
 */
public class SpringBeanListConverter extends DelimitedListConverter<SpringBeanPointer> {

  public SpringBeanListConverter() {
    super(SpringCommonUtils.SPRING_DELIMITERS);
  }

  @Override
  public boolean canResolveTo(Class<? extends PsiElement> elementClass) {
    return ReflectionUtil.isAssignable(SpringBeanPointer.class, elementClass);
  }

  @Nullable
  protected SpringBeanPointer convertString(final @Nullable String s, final ConvertContext context) {
    if (s == null) return null;

    final SpringModel model = SpringConverterUtil.getSpringModel(context);
    if (model == null) return null;

    return SpringModelSearchers.findBean(model, s);
  }

  @Nullable
  protected String toString(@Nullable final SpringBeanPointer springBeanPointer) {
    return springBeanPointer == null ? null : springBeanPointer.getName();
  }

  @Nullable
  protected PsiElement resolveReference(final SpringBeanPointer springBeanPointer,
                                        final ConvertContext context) {
    return springBeanPointer == null ? null : springBeanPointer.getPsiElement();
  }

  protected String getUnresolvedMessage(final String value) {
    return SpringApiBundle.message("model.bean.error.message", value);
  }

  protected Object[] getReferenceVariants(final ConvertContext context,
                                          GenericDomValue<List<SpringBeanPointer>> genericDomValue) {
    final SpringModel model = SpringConverterUtil.getSpringModel(context);
    if (model == null) return EMPTY_ARRAY;

    List<SpringBeanPointer> variants = new ArrayList<>();

    final DomSpringBean currentBean = SpringConverterUtil.getCurrentBean(context);

    final Collection<SpringBeanPointer> allBeans = getVariantBeans(context, model);
    for (SpringBeanPointer pointer : allBeans) {
      if (pointer.isReferenceTo(currentBean)) continue;

      final String beanName = pointer.getName();
      if (beanName != null) {
        for (String string : model.getAllBeanNames(beanName)) {
          if (StringUtil.isNotEmpty(string)) {
            variants.add(pointer.derive(string));
          }
        }
      }
    }

    final List<SpringBeanPointer> existingBeans = genericDomValue.getValue();
    if (existingBeans != null) {
      for (Iterator<SpringBeanPointer> it = variants.iterator(); it.hasNext(); ) {
        final CommonSpringBean variant = it.next().getSpringBean();
        for (SpringBeanPointer existing : existingBeans) {
          if (existing.isReferenceTo(variant)) {
            it.remove();
            break;
          }
        }
      }
    }

    List<LookupElement> result = new ArrayList<>(variants.size());
    for (final SpringBeanPointer pointer : variants) {
      ContainerUtil.addIfNotNull(result, SpringConverterUtil.createCompletionVariant(pointer));
    }

    return result.toArray();
  }

  @SuppressWarnings("MethodMayBeStatic")
  protected Collection<SpringBeanPointer> getVariantBeans(ConvertContext convertContext,
                                                          @NotNull SpringModel model) {
    final List<PsiClassType> requiredBeanTypeClasses = SpringConverterUtil.getRequiredBeanTypeClasses(convertContext);
    if (!requiredBeanTypeClasses.isEmpty()) {
      final CommonSpringBean currentBean = SpringConverterUtil.getCurrentBeanCustomAware(convertContext);
      return SpringConverterUtil.getSmartVariants(currentBean, requiredBeanTypeClasses, model);
    }
    return model.getAllCommonBeans();
  }

  @NotNull
  protected PsiReference createPsiReference(final PsiElement element,
                                            final int start,
                                            final int end,
                                            final ConvertContext context,
                                            final GenericDomValue<List<SpringBeanPointer>> genericDomValue,
                                            final boolean delimitersOnly) {
    return new MyFixableReference(element, getTextRange(genericDomValue, start, end), context, genericDomValue, delimitersOnly);
  }


  private class MyFixableReference extends MyPsiReference implements LocalQuickFixProvider {

    private final Beans beans;

    private MyFixableReference(final PsiElement element,
                               final TextRange range,
                               final ConvertContext context,
                               final GenericDomValue<List<SpringBeanPointer>> genericDomValue,
                               final boolean delimitersOnly) {
      super(element, range, context, genericDomValue, delimitersOnly);
      this.beans = genericDomValue.getParentOfType(Beans.class, false);
    }

    public LocalQuickFix[] getQuickFixes() {
      return SpringBeanResolveQuickFixManager.getInstance()
        .getQuickFixes(myContext, beans, getValue(), SpringConverterUtil.getRequiredBeanTypeClasses(myContext));
    }
  }
}
