/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.utils;

import com.intellij.codeInspection.dataFlow.StringExpressionHelper;
import com.intellij.openapi.util.Pair;
import com.intellij.psi.*;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.LocalSearchScope;
import com.intellij.psi.util.InheritanceUtil;
import com.intellij.psi.util.PropertyUtil;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.psi.xml.XmlAttribute;
import com.intellij.psi.xml.XmlTag;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.jam.javaConfig.ContextJavaBean;
import com.intellij.spring.model.xml.DomSpringBean;
import com.intellij.spring.model.xml.beans.*;
import com.intellij.util.ObjectUtils;
import com.intellij.util.PairProcessor;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.DomElement;
import com.intellij.util.xml.DomUtil;
import com.intellij.util.xml.GenericAttributeValue;
import com.intellij.util.xml.GenericDomValue;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

public class SpringPropertyUtils {

  @Nullable
  public static GenericDomValue<?> getPropertyDomValue(@NotNull SpringValueHolderDefinition valueHolderDefinition) {
    final GenericDomValue<?> valueElement = valueHolderDefinition.getValueElement();
    return valueElement != null && valueElement.getStringValue() == null ? null : valueElement;
  }


  @NotNull
  public static Set<String> getArrayPropertyStringValues(@NotNull CommonSpringBean bean, @NotNull String propertyName) {
    Set<String> set = ContainerUtil.newLinkedHashSet();

    String value = getPropertyStringValue(bean, propertyName);
    if (value != null) {
      set.addAll(SpringCommonUtils.tokenize(value));
    }
    else if (bean instanceof SpringBean) {
      SpringPropertyDefinition propertyDefinition = ((SpringBean)bean).getProperty(propertyName);
      if (propertyDefinition instanceof SpringProperty) {
        set.addAll(getStringValues(((SpringProperty)propertyDefinition).getList()));
        set.addAll(getStringValues(((SpringProperty)propertyDefinition).getSet()));
        set.addAll(getStringValues(((SpringProperty)propertyDefinition).getArray()));
      }
    }

    return set;
  }

  @NotNull
  private static Set<String> getStringValues(@NotNull CollectionElements elements) {
    Set<String> strings = ContainerUtil.newLinkedHashSet();
    for (SpringValue springValue : elements.getValues()) {
      ContainerUtil.addIfNotNull(strings, springValue.getStringValue());
    }
    return strings;
  }

  @Nullable
  public static String getPropertyStringValue(CommonSpringBean bean, String propertyName) {
    if (bean instanceof SpringBean) {
      SpringPropertyDefinition property = ((SpringBean)bean).getProperty(propertyName);
      return property == null ? null : property.getValueAsString();
    }

    // @Bean annotated
    if (bean instanceof ContextJavaBean) {
      final PsiClass beanClass = PsiTypesUtil.getPsiClass(bean.getBeanType());
      if (beanClass != null) {
        final PsiMethod byMethodReturnTypeBeanClass = PropertyUtil.findPropertySetter(beanClass, propertyName, false, true);

        // fallback to "new MyImpl()" in @Bean
        final PsiMethod setter =
          ObjectUtils.chooseNotNull(byMethodReturnTypeBeanClass,
                                    PropertyUtil.findPropertySetter(PsiTypesUtil.getPsiClass(bean.getBeanType(true)), propertyName, false, true));
        if (setter != null) {
          final LocalSearchScope scope = new LocalSearchScope(((ContextJavaBean)bean).getPsiElement());
          final Set<Pair<PsiElement, String>> stringExpressions =
            StringExpressionHelper.searchStringExpressions(setter, scope, 0);
          if (!stringExpressions.isEmpty()) {
            return stringExpressions.iterator().next().second;
          }
        }
      }
    }
    return null;
  }

  @NotNull
  public static Set<String> getListOrSetValues(@NotNull final SpringElementsHolder elementsHolder) {
    if (DomUtil.hasXml(elementsHolder.getList())) {
      return getValues(elementsHolder.getList());
    }
    if (DomUtil.hasXml(elementsHolder.getSet())) {
      return getValues(elementsHolder.getSet());
    }
    if (DomUtil.hasXml(elementsHolder.getArray())) {
      return getValues(elementsHolder.getArray());
    }
    return Collections.emptySet();
  }

  @NotNull
  public static Set<String> getValues(@NotNull final ListOrSet listOrSet) {
    final Set<String> values = ContainerUtil.newLinkedHashSet();
    for (SpringValue value : listOrSet.getValues()) {
      ContainerUtil.addIfNotNull(values, value.getStringValue());
    }
    return values;
  }

  public static PsiArrayType getArrayType(@NotNull ListOrSet array) {
    PsiClass psiClass = array.getValueType().getValue();
    if (psiClass != null) {
      return PsiTypesUtil.getClassType(psiClass).createArrayType();
    }

    GlobalSearchScope scope = GlobalSearchScope.allScope(array.getManager().getProject());
    return PsiType.getJavaLangObject(array.getXmlTag().getManager(), scope).createArrayType();
  }

  public static boolean isSpecificProperty(@NotNull GenericDomValue value, @NotNull String propertyName, @NotNull String... classNames) {
    SpringProperty springProperty = value.getParentOfType(SpringProperty.class, false);
    if (springProperty != null && propertyName.equals(springProperty.getPropertyName())) {
      SpringBean bean = springProperty.getParentOfType(SpringBean.class, false);
      if (bean != null) {
        PsiClass beanClass = PsiTypesUtil.getPsiClass(bean.getBeanType(true));
        if (beanClass == null) {
          return false;
        }

        for (String className : classNames) {
          if (InheritanceUtil.isInheritor(beanClass, className)) {
            return true;
          }
        }
      }
    }

    return false;
  }

  @Nullable
  public static SpringPropertyDefinition findPropertyByName(@NotNull final CommonSpringBean bean,
                                                            @NotNull final String propertyName) {
    return bean instanceof SpringBean ? ((SpringBean)bean).getProperty(propertyName) : null;
  }

  public static List<SpringValueHolderDefinition> getValueHolders(@NotNull CommonSpringBean bean) {
    return bean instanceof DomSpringBean
           ? DomUtil.getDefinedChildrenOfType((DomElement)bean, SpringValueHolderDefinition.class)
           : Collections.emptyList();
  }

  public static List<SpringPropertyDefinition> getProperties(@NotNull CommonSpringBean bean) {
    return bean instanceof DomSpringBean
           ? DomUtil.getDefinedChildrenOfType((DomElement)bean, SpringPropertyDefinition.class)
           : Collections.emptyList();
  }

  public static List<SpringBeanPointer> getSpringValueHolderDependencies(final SpringValueHolderDefinition valueHolder) {
    return new ArrayList<>(getValueHolderDependencies(valueHolder).keySet());
  }

  public static Map<SpringBeanPointer, DomElement> getValueHolderDependencies(final SpringValueHolderDefinition valueHolder) {
    Map<SpringBeanPointer, DomElement> beans = new LinkedHashMap<>();
    addValueHolder(valueHolder, beans);
    return beans;
  }

  private static void addValueHolder(final SpringValueHolderDefinition definition, final Map<SpringBeanPointer, DomElement> beans) {
    final GenericDomValue<SpringBeanPointer> element = definition.getRefElement();
    if (element != null) {
      addBasePointer(element, beans);
    }

    if (definition instanceof SpringValueHolder) {
      final SpringValueHolder valueHolder = (SpringValueHolder)definition;
      addSpringRefBeans(valueHolder.getRef(), beans);
      addIdrefBeans(valueHolder.getIdref(), beans);

      processCollections(beans, valueHolder);

      if (DomUtil.hasXml(valueHolder.getMap())) {
        addMapReferences(valueHolder.getMap(), beans);
      }

      SpringBean innerBean = valueHolder.getBean();
      if (DomUtil.hasXml(innerBean)) {
        beans.put(BeanService.getInstance().createSpringBeanPointer(innerBean), innerBean);
      }
    }
  }

  private static void processCollections(Map<SpringBeanPointer, DomElement> beans, SpringValueHolder valueHolder) {
    final List<ListOrSet> listOrSets = DomUtil.getChildrenOfType(valueHolder, ListOrSet.class);
    for (ListOrSet listOrSet : listOrSets) {
      final DomElement domElement = listOrSet.getManager()
        .getDomElement(listOrSet.getXmlTag()); // instantiate dom element of proper type,  <util:list .../> is UtilList.class

      if (domElement instanceof DomSpringBean) {
        beans.put(BeanService.getInstance().createSpringBeanPointer((DomSpringBean)domElement), domElement);
      }
      else {
        addCollectionReferences(listOrSet, beans);
      }
    }
  }

  private static void addBasePointer(@NotNull final GenericDomValue<SpringBeanPointer> value,
                                     final Map<SpringBeanPointer, DomElement> beans) {
    final SpringBeanPointer beanPointer = value.getValue();
    if (beanPointer == null) {
      return;
    }
    SpringBeanPointer basePointer = beanPointer.getBasePointer();
    beans.put(basePointer, value);
  }

  private static void addMapReferences(final SpringMap map, final Map<SpringBeanPointer, DomElement> beans) {
    for (SpringEntry entry : map.getEntries()) {
      addValueHolder(entry, beans);
    }
  }

  private static void addIdrefBeans(final Idref idref, final Map<SpringBeanPointer, DomElement> beans) {
    addBasePointer(idref.getLocal(), beans);
    addBasePointer(idref.getBean(), beans);
  }

  private static void addSpringRefBeans(final SpringRef springRef, final Map<SpringBeanPointer, DomElement> beans) {
    if (DomUtil.hasXml(springRef)) {
      addBasePointer(springRef.getBean(), beans);
      addBasePointer(springRef.getLocal(), beans);
    }
  }

  public static void addCollectionReferences(final CollectionElements elements, final Map<SpringBeanPointer, DomElement> beans) {
    for (SpringRef springRef : elements.getRefs()) {
      addSpringRefBeans(springRef, beans);
    }
    for (Idref idref : elements.getIdrefs()) {
      addIdrefBeans(idref, beans);
    }
    for (ListOrSet listOrSet : elements.getLists()) {
      addCollectionReferences(listOrSet, beans);
    }
    for (ListOrSet listOrSet : elements.getSets()) {
      addCollectionReferences(listOrSet, beans);
    }
    for (ListOrSet listOrSet : elements.getArrays()) {
      addCollectionReferences(listOrSet, beans);
    }

    for (SpringBean innerBean : elements.getBeans()) {
      beans.put(BeanService.getInstance().createSpringBeanPointer(innerBean), innerBean);
    }
    for (SpringMap map : elements.getMaps()) {
      addMapReferences(map, beans);
    }
  }

  public static List<SpringBeanPointer> getCollectionElementDependencies(final CollectionElements collectionElements) {
    Map<SpringBeanPointer, DomElement> beans = new LinkedHashMap<>();
    addCollectionReferences(collectionElements, beans);
    return new ArrayList<>(beans.keySet());
  }

  public static boolean processSpringValues(final SpringProperty property,
                                            final PairProcessor<GenericDomValue, String> processor) {
    final GenericAttributeValue<String> valueAttr = property.getValueAttr();
    final XmlAttribute valueAttrElement = valueAttr.getXmlAttribute();
    final String valueAttrString = valueAttr.getStringValue();
    if (valueAttrElement != null && valueAttrString != null && !processor.process(valueAttr, valueAttrString)) {
      return false;
    }

    final SpringValue value = property.getValue();
    final XmlTag valueElement = value.getXmlTag();
    final String valueString = value.getStringValue();
    if (valueElement != null && valueString != null && !processor.process(value, valueString)) {
      return false;
    }

    if (!processSpringListOrSetValues(property.getList(), processor)) return false;
    if (!processSpringListOrSetValues(property.getSet(), processor)) return false;
    if (!processSpringListOrSetValues(property.getArray(), processor)) return false;
    if (!processSpringPointerValue(property.getRefAttr(), processor)) return false;
    if (!processSpringPointerValue(property.getRef().getBean(), processor)) return false;
    return true;
  }

  private static boolean processSpringPointerValue(GenericAttributeValue<SpringBeanPointer> pointerValue,
                                                   PairProcessor<GenericDomValue, String> processor) {
    final SpringBeanPointer pointer = pointerValue.getValue();
    final CommonSpringBean bean = pointer == null ? null : pointer.getSpringBean();
    if (bean instanceof ListOrSet) {
      if (!processSpringListOrSetValues((ListOrSet)bean, processor)) return false;
    }
    else if (bean != null) {
      SpringPropertyDefinition value = findPropertyByName(bean, "sourceList");
      if (value == null) value = findPropertyByName(bean, "sourceSet");
      if (value instanceof SpringProperty) {
        if (!processSpringListOrSetValues(((SpringProperty)value).getList(), processor)) return false;
      }
    }
    return true;
  }

  private static boolean processSpringListOrSetValues(ListOrSet listOrSet,
                                                      PairProcessor<GenericDomValue, String> processor) {
    for (SpringValue springValue : listOrSet.getValues()) {
      final XmlTag element = springValue.getXmlTag();
      final String string = springValue.getStringValue();
      if (element != null && string != null && !processor.process(springValue, string)) {
        return false;
      }
    }
    return true;
  }

  @Nullable
  public static SpringBeanPointer findReferencedBean(@NotNull SpringPropertyDefinition definition) {
    final SpringBeanPointer springBeanPointer = definition.getRefValue();
    if (springBeanPointer != null) {
      return springBeanPointer;
    }
    return definition instanceof SpringInjection ? findReferencedBean((SpringInjection)definition) : null;
  }

  @Nullable
  public static SpringBeanPointer findReferencedBean(@NotNull SpringInjection injection) {
    final SpringBeanPointer refAttrPointer = injection.getRefAttr().getValue();
    if (refAttrPointer != null) {
      return refAttrPointer;
    }
    if (DomUtil.hasXml(injection.getRef())) {
      final SpringRef springRef = injection.getRef();

      final SpringBeanPointer beanPointer = springRef.getBean().getValue();
      if (beanPointer != null) {
        return beanPointer;
      }
      final SpringBeanPointer localPointer = springRef.getLocal().getValue();
      if (localPointer != null) {
        return localPointer;
      }
      final SpringBeanPointer parentPointer = springRef.getParentAttr().getValue();
      if (parentPointer != null) {
        return parentPointer;
      }
    }
    else if (DomUtil.hasXml(injection.getBean())) {
      return BeanService.getInstance().createSpringBeanPointer(injection.getBean());
    }

    return null;
  }
}
