const fs:typeof NodeJS.fs = require("fs");

export const isLogEnabled = process.env["TSS_LOG"];


class LoggerImpl implements ts.server.Logger {
    fd = -1;
    seq = 0;
    inGroup = false;
    firstInGroup = true;

    constructor(public logFilename:string, public level:string) {
    }

    static padStringRight(str:string, padding:string) {
        return (str + padding).slice(0, padding.length);
    }

    close() {
        if (this.fd >= 0) {
            fs.close(this.fd);
        }
    }

    perftrc(s:string) {
        this.msg(s, "Perf");
    }

    info(s:string) {
        this.msg(s, "Info");
    }

    startGroup() {
        this.inGroup = true;
        this.firstInGroup = true;
    }

    endGroup() {
        this.inGroup = false;
        this.seq++;
        this.firstInGroup = true;
    }

    loggingEnabled() {
        return !!this.logFilename;
    }

    isVerbose() {
        return this.loggingEnabled() && (this.level == "verbose");
    }


    msg(s:string, type = "Err") {
        if (this.fd < 0) {
            if (this.logFilename) {
                try {
                    this.fd = fs.openSync(this.logFilename, "w");
                } catch (e) {
                    serverLogger(e.message + " " + e.stack);
                    this.logFilename = null;
                }
            }
        }
        if (this.fd >= 0) {
            s = s + "\n";
            const prefix = LoggerImpl.padStringRight(type + " " + this.seq.toString(), "          ");
            if (this.firstInGroup) {
                s = prefix + s;
                this.firstInGroup = false;
            }
            if (!this.inGroup) {
                this.seq++;
                this.firstInGroup = true;
            }
            const buf = new Buffer(s);
            fs.writeSync(this.fd, buf, 0, buf.length, null);
        }
    }
}

interface LogOptions {
    file?:string;
    detailLevel?:string;
}

function parseLoggingEnvironmentString(logEnvStr:string):LogOptions {
    const logEnv:LogOptions = {};
    const args = logEnvStr.split(" ");
    for (let i = 0, len = args.length; i < (len - 1); i += 2) {
        const option = args[i];
        const value = args[i + 1];
        if (option && value) {
            switch (option) {
                case "-file":
                    logEnv.file = value;
                    break;
                case "-level":
                    logEnv.detailLevel = value;
                    break;
            }
        }
    }
    return logEnv;
}

export function createLoggerFromEnv() {
    let fileName:string = undefined;
    let detailLevel = "normal";
    if (isLogEnabled) {
        try {
            const logEnv = parseLoggingEnvironmentString(isLogEnabled);
            if (logEnv.file) {
                fileName = logEnv.file;
            }
            else {
                fileName = process.cwd() + "/.log" + process.pid.toString();
            }
            if (logEnv.detailLevel) {
                detailLevel = logEnv.detailLevel;
            }
        } catch (e) {
            serverLogger(e.message + " " + e.stack, true);
        }
    }
    return new LoggerImpl(fileName, detailLevel);
}



export function serverLogger(message, force?) {
    if (isLogEnabled || force) {
        console.error("Process: " + message);
    }
}