/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.contexts.model.graph;

import com.intellij.psi.PsiElement;
import com.intellij.psi.impl.FakePsiElement;
import com.intellij.psi.xml.XmlElement;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.xml.DomElement;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * @since 14.1
 */
public class LocalModelDependency {

  private final LocalModelDependencyType myType;
  private final PsiElement myIdentifyingElement;
  private final String myLabel;

  public static LocalModelDependency create() {
    return new LocalModelDependency("", LocalModelDependencyType.UNKNOWN, null);
  }

  public static LocalModelDependency create(@NotNull LocalModelDependencyType type,
                                            @NotNull PsiElement identifyingElement) {
    return new LocalModelDependency(identifyingElement.getText(), type, identifyingElement);
  }

  /**
   * Create a dependency defined by {@link DomElement} avoiding PSI access.
   *
   * @since 2016.3
   */
  public static LocalModelDependency create(@NotNull String label,
                                            @NotNull LocalModelDependencyType type,
                                            DomElement identifyingElement) {
    return new LocalModelDependency(label, type, new FakePsiElement() {

      @NotNull
      private XmlElement getXmlElement() {
        final XmlElement xmlElement = identifyingElement.getXmlElement();
        assert xmlElement != null : identifyingElement;
        return xmlElement;
      }

      @Override
      public PsiElement getParent() {
        return getXmlElement();
      }

      // for diagram
      @Override
      public void delete() throws IncorrectOperationException {
        getXmlElement().delete();
      }

      @NotNull
      @Override
      public PsiElement getNavigationElement() {
        return getXmlElement();
      }
    });
  }

  public static LocalModelDependency create(@NotNull String label,
                                            @NotNull LocalModelDependencyType type,
                                            @Nullable PsiElement identifyingElement) {
    return new LocalModelDependency(label, type, identifyingElement);
  }

  private LocalModelDependency(@NotNull String label,
                               @NotNull LocalModelDependencyType type,
                               @Nullable PsiElement identifyingElement) {
    myLabel = label;
    myType = type;
    myIdentifyingElement = identifyingElement;
  }

  @NotNull
  public LocalModelDependencyType getType() {
    return myType;
  }

  @Nullable
  public PsiElement getIdentifyingElement() {
    return myIdentifyingElement;
  }

  @Override
  public String toString() {
    return "LocalModelDependency{" + myType + ", " + myIdentifyingElement + '}';
  }

  public String getLabel() {
    return myLabel;
  }
}
