/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.testContexts;

import com.intellij.jam.JamClassAttributeElement;
import com.intellij.jam.JamConverter;
import com.intellij.jam.JamService;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamAttributeMeta;
import com.intellij.jam.reflect.JamClassAttributeMeta;
import com.intellij.jam.reflect.JamMemberMeta;
import com.intellij.openapi.util.NullableLazyValue;
import com.intellij.psi.PsiAnchor;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.util.PsiTreeUtil;
import com.intellij.psi.xml.XmlFile;
import com.intellij.semantic.SemKey;
import com.intellij.spring.constants.SpringAnnotationsConstants;
import com.intellij.spring.model.aliasFor.SpringAliasFor;
import com.intellij.spring.model.aliasFor.SpringAliasForUtils;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;
import java.util.Set;

/**
 * @since 16
 */
public class CustomContextConfiguration implements ContextConfiguration {

  public static final SemKey<JamMemberMeta<PsiClass, CustomContextConfiguration>> META_KEY =
    SpringContextConfiguration.META.getMetaKey().subKey("CustomContextConfiguration");
  public static final SemKey<CustomContextConfiguration> JAM_KEY =
    ContextConfiguration.CONTEXT_CONFIGURATION_JAM_KEY.subKey("CustomContextConfiguration");

  protected final PsiElementRef<PsiAnnotation> myPsiAnnotation;

  protected final JamAnnotationMeta myAnnotationMeta;
  private final PsiAnchor myPsiClassAnchor;

  private final NullableLazyValue<SpringContextConfiguration> myDefiningMetaAnnotation =
    new NullableLazyValue<SpringContextConfiguration>() {
      @Nullable
      @Override
      protected SpringContextConfiguration compute() {
        final PsiAnnotation definingMetaAnnotation = SpringAliasForUtils
          .findDefiningMetaAnnotation(getPsiElement(), myAnnotationMeta.getAnnoName(),
                                      SpringAnnotationsConstants.CONTEXT_CONFIGURATION);
        if (definingMetaAnnotation != null) {
          final PsiClass annotationType = PsiTreeUtil.getParentOfType(definingMetaAnnotation, PsiClass.class, true);
          if (annotationType != null) {
            return JamService.getJamService(getPsiElement().getProject()).getJamElement(annotationType, SpringContextConfiguration.META);
          }
        }
        return null;
      }
    };

  public CustomContextConfiguration(@NotNull String anno, @NotNull PsiClass psiClassAnchor) {
    this(new JamAnnotationMeta(anno), psiClassAnchor);
  }

  protected CustomContextConfiguration(@NotNull JamAnnotationMeta annotationMeta, @NotNull PsiClass psiClassAnchor) {
    myAnnotationMeta = annotationMeta;
    myPsiClassAnchor = PsiAnchor.create(psiClassAnchor);
    myPsiAnnotation = myAnnotationMeta.getAnnotationRef(getPsiElement());
  }

  @Override
  public PsiClass getPsiElement() {
    return (PsiClass)myPsiClassAnchor.retrieve();
  }

  @Nullable
  public PsiAnnotation getAnnotation() {
    return myPsiAnnotation.getPsiElement();
  }

  @Override
  @NotNull
  public Set<XmlFile> getLocations(@NotNull PsiClass... contexts) {
    Set<XmlFile> files = ContainerUtil.newLinkedHashSet();

    JamConverter<List<XmlFile>> referenceConverter = new ApplicationContextReferenceConverter();
    for (String attrName : XML_FILES_ATTRS) {
      SpringAliasFor aliasFor = getAliasAttribute(attrName);
      if (aliasFor != null) {
        JamStringAttributeElement<List<XmlFile>> xmlFiles =
          new JamStringAttributeElement<>(myPsiAnnotation, getAliasForMethodName(aliasFor), referenceConverter);
        List<XmlFile> values = xmlFiles.getValue();
        if (values != null) files.addAll(values);
        return files;
      }
    }

    final ContextConfiguration definingContextConfiguration = getDefiningContextConfiguration();
    if (definingContextConfiguration != null) {
      return definingContextConfiguration.getLocations(definingContextConfiguration.getPsiElement());
    }

    return Collections.emptySet();
  }

  @Override
  @NotNull
  public List<PsiClass> getConfigurationClasses() {
    List<PsiClass> psiClasses = ContainerUtil.newSmartList();
    SpringAliasFor aliasFor = getAliasAttribute(CLASSES_ATTR_NAME);
    if (aliasFor != null) {
      JamClassAttributeMeta.Collection collection = new JamClassAttributeMeta.Collection(getAliasForMethodName(aliasFor));
      for (JamClassAttributeElement classAttributeElement : collection.getJam(myPsiAnnotation)) {
        ContainerUtil.addIfNotNull(psiClasses, classAttributeElement.getValue());
      }
      return psiClasses;
    }

    final ContextConfiguration definingContextConfiguration = getDefiningContextConfiguration();
    if (definingContextConfiguration != null) {
      return definingContextConfiguration.getConfigurationClasses();
    }

    return Collections.emptyList();
  }

  private ContextConfiguration getDefiningContextConfiguration() {
    return myDefiningMetaAnnotation.getValue();
  }

  @NotNull
  private static String getAliasForMethodName(SpringAliasFor aliasFor) {
    return aliasFor.getPsiElement().getName();
  }

  @Override
  public boolean hasLocationsAttribute() {
    final SpringAliasFor aliasAttribute = getAliasAttribute(LOCATIONS_ATTR_NAME);
    if (aliasAttribute != null) return true;

    final ContextConfiguration definingContextConfiguration = getDefiningContextConfiguration();
    return definingContextConfiguration != null && definingContextConfiguration.hasLocationsAttribute();
  }

  @Override
  public boolean hasValueAttribute() {
    final SpringAliasFor aliasAttribute = getAliasAttribute(VALUE_ATTR_NAME);
    if (aliasAttribute != null) return true;

    final ContextConfiguration definingContextConfiguration = getDefiningContextConfiguration();
    return definingContextConfiguration != null && definingContextConfiguration.hasValueAttribute();
  }

  @Nullable
  @Override
  public PsiClass getLoaderClass() {
    SpringAliasFor aliasFor = getAliasAttribute(LOADER_ATTR_NAME);
    if (aliasFor != null) {
      return JamAttributeMeta.singleClass(getAliasForMethodName(aliasFor)).getJam(myPsiAnnotation).getValue();
    }
    return null;
  }

  private SpringAliasFor getAliasAttribute(@NotNull String attrName) {
    return SpringAliasForUtils.findAliasFor(getPsiElement(),
                                            myAnnotationMeta.getAnnoName(),
                                            SpringAnnotationsConstants.CONTEXT_CONFIGURATION,
                                            attrName);
  }
}