/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model;

import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiSubstitutor;
import com.intellij.psi.PsiType;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.spring.model.utils.SpringCommonUtils;
import org.jetbrains.annotations.NotNull;

public abstract class SpringModelSearchParameters {

  protected SpringModelSearchParameters() {
  }

  /**
   * Returns whether given search parameters are valid to start a search.
   *
   * @return {@code false} if search cannot be performed.
   */
  public abstract boolean canSearch();

  public static BeanClass byClass(@NotNull PsiClass psiClass) {
    return new BeanClass(psiClass);
  }

  /**
   * @since 2016.2
   */
  public static BeanClass byType(@NotNull PsiType psiType) {
    return new BeanClass(psiType);
  }

  public static class BeanClass extends SpringModelSearchParameters {
    private final PsiType myType;

    private final boolean myCanSearch;

    private boolean myWithInheritors;

    private boolean myEffectiveBeanTypes;

    private BeanClass(@NotNull PsiClass psiClass) {
      myType = JavaPsiFacade.getElementFactory(psiClass.getProject()).createType(psiClass, PsiSubstitutor.EMPTY);
      myCanSearch = SpringCommonUtils.isSpringBeanCandidateClass(psiClass);
    }

    private BeanClass(@NotNull PsiType psiType) {
      myType = psiType;
      final PsiClass searchClass = PsiTypesUtil.getPsiClass(psiType);
      myCanSearch = searchClass == null || SpringCommonUtils.isSpringBeanCandidateClass(searchClass);
    }

    @Override
    public boolean canSearch() {
      return myCanSearch;
    }

    public BeanClass withInheritors() {
      myWithInheritors = true;
      return this;
    }

    /**
     * NB: void, no effect since {@code 2016.1}.
     * <p/>
     * Use with library classes search to avoid using their default use scope.
     *
     * @param inheritorsSearchScope Search scope for searching inheritors, e.g. module.
     * @return This.
     * @since 15
     * @deprecated Use {@link #withInheritors()}. To remove in {@code 2016.3}.
     */
    @Deprecated
    public BeanClass withInheritors(GlobalSearchScope inheritorsSearchScope) {
      return withInheritors();
    }

    public BeanClass effectiveBeanTypes() {
      myEffectiveBeanTypes = true;
      return this;
    }

    @NotNull
    public PsiType getSearchType() {
      return myType;
    }

    public boolean isWithInheritors() {
      return myWithInheritors;
    }

    public boolean isEffectiveBeanTypes() {
      return myEffectiveBeanTypes;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (!(o instanceof BeanClass)) return false;

      BeanClass beanClass = (BeanClass)o;

      if (myEffectiveBeanTypes != beanClass.myEffectiveBeanTypes) return false;
      if (myWithInheritors != beanClass.myWithInheritors) return false;
      if (myCanSearch != beanClass.myCanSearch) return false;
      if (!myType.isValid() || !beanClass.getSearchType().isValid()) return false;
      if (!myType.equals(beanClass.getSearchType())) return false;
      return true;
    }

    @Override
    public int hashCode() {
      int result = myType.hashCode();
      result = 31 * result + (myWithInheritors ? 1 : 0);
      result = 31 * result + (myEffectiveBeanTypes ? 1 : 0);
      return result;
    }
  }

  public static BeanName byName(@NotNull String beanName) {
    return new BeanName(beanName);
  }

  public static class BeanName extends SpringModelSearchParameters {

    @NotNull
    private final String myBeanName;
    private final boolean myCanSearch;

    private BeanName(@NotNull String beanName) {
      myBeanName = beanName;
      myCanSearch = StringUtil.isNotEmpty(beanName);
    }

    @Override
    public boolean canSearch() {
      return myCanSearch;
    }

    @NotNull
    public String getBeanName() {
      return myBeanName;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (!(o instanceof BeanName)) return false;

      BeanName name = (BeanName)o;

      if (myCanSearch != name.myCanSearch) return false;
      if (!myBeanName.equals(name.myBeanName)) return false;

      return true;
    }

    @Override
    public int hashCode() {
      return myBeanName.hashCode();
    }
  }
}
