"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var logger_impl_1 = require("./logger-impl");
var compile_info_holder_1 = require("./compile-info-holder");
var ts15impl_1 = require("./ts15impl");
var util_1 = require("./util");
/**
 * Default tsserver implementation doesn't return response in most cases ("open", "close", etc.)
 * we want to override the behaviour and send empty-response holder
 */
var doneRequest = {
    responseRequired: true,
    response: "done"
};
var DETAILED_COMPLETION_COUNT = 5;
function getSession(ts_impl /*must be typeof ts */, logger, commonDefaultOptions, pathProcessor, mainFile) {
    var TypeScriptSession = ts_impl.server.Session;
    var TypeScriptProjectService = ts_impl.server.ProjectService;
    var TypeScriptCommandNames = ts_impl.server.CommandNames;
    util_1.initCommandNames(TypeScriptCommandNames);
    var isVersionTypeScript15 = util_1.isTypeScript15(ts_impl);
    var isVersionTypeScript16 = util_1.isTypeScript16(ts_impl);
    if (isVersionTypeScript15) {
        ts15impl_1.setGetFileNames(ts_impl.server.Project);
    }
    // var 
    var IDESession = (function (_super) {
        __extends(IDESession, _super);
        function IDESession(host, byteLength, hrtime, logger) {
            if (isVersionTypeScript15) {
                _super.call(this, host, logger, undefined, undefined);
            }
            else {
                _super.call(this, host, byteLength, hrtime, logger);
            }
            this._host = host;
            this._wasFirstMessage = false;
            var handler = this.projectService.eventHandler;
            //reuse handler
            this.projectService = new IDEProjectService(host, logger, handler);
        }
        IDESession.prototype.send = function (msg) {
            var json = JSON.stringify(msg);
            this._host.write(json + "\n");
        };
        IDESession.prototype.onMessage = function (message) {
            if (isVersionTypeScript15) {
                ts15impl_1.onMessage15(this, message);
                return;
            }
            _super.prototype.onMessage.call(this, message);
        };
        IDESession.prototype.executeCommand = function (request) {
            var startTime = Date.now();
            var command = request.command;
            try {
                if (TypeScriptCommandNames.Open == command) {
                    //use own implementation
                    var openArgs = request.arguments;
                    this.openClientFileExt(openArgs);
                    return doneRequest;
                }
                else if (TypeScriptCommandNames.ReloadProjects == command) {
                    this.getIDEProjectService().projectEmittedWithAllFiles = {};
                    if (isVersionTypeScript15)
                        return ts15impl_1.reload15(this, ts_impl);
                    this.updateProjectStructureExt();
                    return doneRequest;
                }
                else if (TypeScriptCommandNames.IDEChangeFiles == command) {
                    var updateFilesArgs = request.arguments;
                    return this.updateFilesExt(updateFilesArgs);
                }
                else if (TypeScriptCommandNames.IDECompile == command) {
                    var fileArgs = request.arguments;
                    return this.compileFileExt(fileArgs);
                }
                else if (TypeScriptCommandNames.Close == command) {
                    if (isVersionTypeScript15) {
                        ts15impl_1.close15(this, request);
                        return doneRequest;
                    }
                    _super.prototype.executeCommand.call(this, request);
                    return doneRequest;
                }
                else if (TypeScriptCommandNames.IDEGetErrors == command) {
                    var args = request.arguments;
                    return { response: { infos: this.getDiagnosticsExt(args.files) }, responseRequired: true };
                }
                else if (TypeScriptCommandNames.IDEGetMainFileErrors == command) {
                    var args = request.arguments;
                    return { response: { infos: this.getMainFileDiagnosticsForFileExt(args.file) }, responseRequired: true };
                }
                else if (TypeScriptCommandNames.IDEGetProjectErrors == command) {
                    var args = request.arguments;
                    return { response: { infos: this.getProjectDiagnosticsForFileExt(args.file) }, responseRequired: true };
                }
                else if (TypeScriptCommandNames.IDECompletions == command) {
                    if (isVersionTypeScript15)
                        return { response: [], responseRequired: true };
                    return this.getCompletionExt(request);
                }
                return _super.prototype.executeCommand.call(this, request);
            }
            finally {
                if (!this._wasFirstMessage) {
                    logger_impl_1.serverLogger("TypeScript service version: " + ts_impl.version);
                    this._wasFirstMessage = true;
                }
                var processingTime = Date.now() - startTime;
                logger_impl_1.serverLogger("Message " + request.seq + " '" + command + "' server time, mills: " + processingTime, true);
            }
        };
        IDESession.prototype.updateFilesExt = function (args) {
            var updated = false;
            var files = args.files;
            for (var fileName in files) {
                if (files.hasOwnProperty(fileName)) {
                    var content = files[fileName];
                    if (content !== undefined) {
                        this.changeFileExt(fileName, content);
                        updated = true;
                    }
                }
            }
            if (args.filesToReloadContentFromDisk) {
                for (var _i = 0, _a = args.filesToReloadContentFromDisk; _i < _a.length; _i++) {
                    var fileName = _a[_i];
                    if (!fileName) {
                        continue;
                    }
                    var file = ts_impl.normalizePath(fileName);
                    this.projectService.closeClientFile(file);
                    if (logger_impl_1.isLogEnabled) {
                        logger_impl_1.serverLogger("Update file from disk (by 'filesToReloadContentFromDisk') " + file);
                    }
                    updated = true;
                }
            }
            if (updated) {
                this.updateProjectStructureExt();
            }
            return doneRequest;
        };
        IDESession.prototype.updateProjectStructureExt = function () {
            var _this = this;
            var mySeq = this.getChangeSeq();
            var matchSeq = function (n) { return n === mySeq; };
            setTimeout(function () {
                if (matchSeq(_this.getChangeSeq())) {
                    var startTime = Date.now();
                    _this.projectService.updateProjectStructure();
                    logger_impl_1.serverLogger("Update project structure scheduler time, mills: " + (Date.now() - startTime), true);
                }
            }, 1500);
        };
        IDESession.prototype.getChangeSeq = function () {
            var anyThis = this;
            var superClassSeq = anyThis.changeSeq;
            if (typeof superClassSeq !== "undefined") {
                return superClassSeq;
            }
            logger_impl_1.serverLogger("WARN: Used own sequence implementation (can be slow)", true);
            return this._mySeq;
        };
        IDESession.prototype.openClientFileExt = function (openArgs) {
            var fileName = openArgs.file;
            var fileContent = openArgs.fileContent;
            var configFile = openArgs.tsConfig;
            var file = ts_impl.normalizePath(fileName);
            return this.projectService.openClientFileExt(file, fileContent, configFile);
        };
        IDESession.prototype.changeFileExt = function (fileName, content, tsconfig) {
            var file = ts_impl.normalizePath(fileName);
            var project = this.projectService.getProjectForFile(file);
            if (project) {
                var compilerService = project.compilerService;
                var scriptInfo = compilerService.host.getScriptInfo(file);
                if (scriptInfo != null) {
                    scriptInfo.svc.reload(content);
                    if (logger_impl_1.isLogEnabled) {
                        logger_impl_1.serverLogger("Update file reload content from text " + file);
                    }
                }
                else {
                    if (logger_impl_1.isLogEnabled) {
                        logger_impl_1.serverLogger("Update file scriptInfo is null " + file);
                    }
                }
            }
            else {
                if (logger_impl_1.isLogEnabled) {
                    logger_impl_1.serverLogger("Update file cannot find project for " + file);
                }
                this.openClientFileExt({
                    file: fileName,
                    fileContent: content,
                    tsConfig: tsconfig
                });
            }
        };
        IDESession.prototype.compileFileExt = function (req) {
            var _this = this;
            var startCompile = Date.now();
            var compileExactFile = req.file != null;
            if (!compileExactFile && !req.tsConfig) {
                return doneRequest;
            }
            var requestedFile = ts_impl.normalizePath(req.file ? req.file : req.tsConfig);
            if (!requestedFile) {
                return doneRequest;
            }
            var project = this.getProjectForCompileRequest(req, requestedFile);
            if (logger_impl_1.isLogEnabled) {
                logger_impl_1.serverLogger("Compile get project end time: " + (Date.now() - startCompile));
            }
            var outFiles = [];
            var includeErrors = req.includeErrors;
            var diagnostics = includeErrors ? [] : undefined;
            var needCompile = (req.force || !project || !project.projectOptions || project.projectOptions.compileOnSave || project.projectOptions.compileOnSave === undefined);
            if (project && needCompile) {
                var projectFilename = project.projectFilename;
                var languageService = project.compilerService.languageService;
                var program_1 = languageService.getProgram();
                if (logger_impl_1.isLogEnabled) {
                    logger_impl_1.serverLogger("Compile get source files end time: " + program_1.getSourceFiles().length + "(count): time, mills: " + (Date.now() - startCompile));
                    logger_impl_1.serverLogger("Compile project Filename: " + (projectFilename ? projectFilename : "no filename"));
                }
                var options_1 = project.projectOptions ? project.projectOptions.compilerOptions : null;
                project.compilerService.host.getNewLine = function () {
                    return ts_impl.getNewLineCharacter(options_1 ? options_1 : {});
                };
                var compileInfoHolder_1 = null;
                if (projectFilename) {
                    compileInfoHolder_1 = this.getIDEProjectService().projectEmittedWithAllFiles[projectFilename];
                    compileExactFile = compileExactFile && compileInfoHolder_1 == null;
                    if (!compileInfoHolder_1) {
                        compileInfoHolder_1 = new compile_info_holder_1.CompileInfoHolder(ts_impl);
                        this.getIDEProjectService().projectEmittedWithAllFiles[projectFilename] = compileInfoHolder_1;
                    }
                }
                else {
                    compileExactFile = false;
                }
                var fileWriteCallback_1 = this.getFileWrite(project, outFiles, req.contentRootForMacro, req.sourceRootForMacro);
                if (!compileExactFile) {
                    logger_impl_1.serverLogger("Compile all files using cache checking", true);
                    var toUpdateFiles_1 = [];
                    var rawSourceFiles = program_1.getSourceFiles();
                    rawSourceFiles.forEach(function (val) {
                        if (!compileInfoHolder_1 || compileInfoHolder_1.checkUpdateAndAddToCache(val)) {
                            toUpdateFiles_1.push(val);
                        }
                    });
                    var compilerOptions = program_1.getCompilerOptions();
                    var useOutFile = compilerOptions && (compilerOptions.outFile || compilerOptions.out);
                    if (toUpdateFiles_1.length > 0) {
                        if (toUpdateFiles_1.length == rawSourceFiles.length || useOutFile) {
                            var emitResult = program_1.emit(undefined, fileWriteCallback_1);
                            diagnostics = this.appendEmitDiagnostics(project, emitResult, diagnostics);
                        }
                        else {
                            toUpdateFiles_1.forEach(function (el) {
                                var emitResult = program_1.emit(el, fileWriteCallback_1);
                                diagnostics = _this.appendEmitDiagnostics(project, emitResult, diagnostics);
                            });
                        }
                    }
                    logger_impl_1.serverLogger("Compile end emit files: " + (Date.now() - startCompile));
                }
                else {
                    var sourceFile = program_1.getSourceFile(requestedFile);
                    if (sourceFile) {
                        if (!compileInfoHolder_1 || compileInfoHolder_1.checkUpdateAndAddToCache(sourceFile)) {
                            var emitResult = project.program.emit(sourceFile, fileWriteCallback_1);
                            diagnostics = this.appendEmitDiagnostics(project, emitResult, diagnostics);
                        }
                    }
                    else {
                        logger_impl_1.serverLogger("Compile can't find source file: shouldn't be happened");
                    }
                }
            }
            else {
                if (project) {
                    logger_impl_1.serverLogger("Compile skip: compileOnSave = false", true);
                }
                else {
                    logger_impl_1.serverLogger("Compile can't find project: shouldn't be happened", true);
                }
            }
            if (includeErrors) {
                diagnostics = diagnostics.concat(compileExactFile ?
                    this.getDiagnosticsExt([requestedFile], project) :
                    this.getProjectDiagnosticsExt(project));
            }
            if (logger_impl_1.isLogEnabled) {
                logger_impl_1.serverLogger("Compile end get diagnostics time, mills: " + (Date.now() - startCompile));
            }
            return { response: { generatedFiles: outFiles, infos: diagnostics }, responseRequired: true };
        };
        IDESession.prototype.getProjectForCompileRequest = function (req, normalizedRequestedFile) {
            var project = null;
            if (req.file) {
                project = this.projectService.getProjectForFile(normalizedRequestedFile);
            }
            else {
                if (isVersionTypeScript15)
                    return ts15impl_1.openProjectByConfig(this.projectService, normalizedRequestedFile, ts_impl);
                project = this.projectService.findConfiguredProjectByConfigFile(normalizedRequestedFile);
                if (!project) {
                    var configResult = this.projectService.openConfigFile(normalizedRequestedFile);
                    if (configResult && configResult.project) {
                        return configResult.project;
                    }
                }
            }
            return project;
        };
        IDESession.prototype.getIDEProjectService = function () {
            return this.projectService;
        };
        IDESession.prototype.getTime = function () {
            return Date.now();
        };
        IDESession.prototype.appendEmitDiagnostics = function (project, emitResult, diagnostics) {
            if (diagnostics !== undefined && emitResult && emitResult.diagnostics) {
                var emitDiagnostics = emitResult.diagnostics;
                return diagnostics.concat(emitDiagnostics.map(function (el) {
                    return { file: el.file.fileName, diagnostics: [formatDiagnostic(el.file.fileName, project, el)] };
                }));
            }
            return diagnostics;
        };
        IDESession.prototype.getFileWrite = function (project, outFiles, contentRoot, sourceRoot) {
            var _this = this;
            return function (fileName, data, writeByteOrderMark, onError, sourceFiles) {
                var normalizedName = ts_impl.normalizePath(fileName);
                normalizedName = fixNameWithProcessor(normalizedName, onError, contentRoot, sourceRoot);
                _this.ensureDirectoriesExist(ts_impl.getDirectoryPath(normalizedName));
                if (logger_impl_1.isLogEnabled) {
                    logger_impl_1.serverLogger("Compile write file: " + normalizedName);
                }
                _this._host.writeFile(normalizedName, data, writeByteOrderMark, onError, sourceFiles);
                outFiles.push(normalizedName);
            };
        };
        IDESession.prototype.logError = function (err, cmd) {
            var typedErr = err;
            logger_impl_1.serverLogger("Error processing message: " + err.message + " " + typedErr.stack, true);
            _super.prototype.logError.call(this, err, cmd);
        };
        IDESession.prototype.getCompletionExt = function (request) {
            var startDate = -1;
            if (logger_impl_1.isLogEnabled) {
                startDate = Date.now();
            }
            var args = request.arguments;
            var result = _super.prototype.executeCommand.call(this, {
                command: TypeScriptCommandNames.Completions,
                arguments: args,
                seq: request.seq,
                type: request.type
            });
            if (logger_impl_1.isLogEnabled) {
                logger_impl_1.serverLogger("Completion service implementation time, mills: " + (Date.now() - startDate));
            }
            var response = result.response;
            var ideCompletions = this.getDetailedIDECompletions(args, response);
            if (logger_impl_1.isLogEnabled) {
                logger_impl_1.serverLogger("Completion with detailed items time, mills: " + (Date.now() - startDate));
            }
            return {
                response: ideCompletions,
                responseRequired: true
            };
        };
        IDESession.prototype.getDetailedIDECompletions = function (req, entries) {
            if (!entries) {
                return entries;
            }
            var file = ts_impl.normalizePath(req.file);
            var project = this.projectService.getProjectForFile(file);
            if (!project) {
                logger_impl_1.serverLogger("Can't find project: shouldn't be happened", true);
                return entries;
            }
            var compilerService = project.compilerService;
            var position = compilerService.host.lineOffsetToPosition(file, req.line, req.offset);
            var count = 0;
            return entries.reduce(function (accum, entry) {
                if (!util_1.isFunctionKind(entry.kind) || count++ > DETAILED_COMPLETION_COUNT) {
                    accum.push(entry);
                }
                else {
                    var details = compilerService.languageService.getCompletionEntryDetails(file, position, entry.name);
                    if (details) {
                        details.sortText = entry.sortText;
                        accum.push(details);
                    }
                }
                return accum;
            }, []);
        };
        /**
         * Possible we can remove the implementation if we will use 'pull' events
         * now just for test we use 'blocking' implementation
         * to check speed of processing
         * todo use 'pull' implementation
         */
        IDESession.prototype.getDiagnosticsExt = function (fileNames, commonProject, reqOpen) {
            var _this = this;
            if (reqOpen === void 0) { reqOpen = true; }
            var projectsToProcess = {};
            var hasEmptyProject = false;
            if (commonProject) {
                if (commonProject.projectFilename) {
                    projectsToProcess[commonProject.projectFilename] = commonProject;
                }
                else {
                    hasEmptyProject = true;
                }
            }
            var checkList = fileNames.reduce(function (accumulator, fileName) {
                fileName = ts_impl.normalizePath(fileName);
                if (commonProject) {
                    accumulator.push({ fileName: fileName, project: commonProject });
                }
                else {
                    var project = _this.projectService.getProjectForFile(fileName);
                    if (project) {
                        accumulator.push({ fileName: fileName, project: project });
                        if (project.projectFilename) {
                            projectsToProcess[project.projectFilename] = project;
                        }
                        else {
                            hasEmptyProject = true;
                        }
                    }
                }
                return accumulator;
            }, []);
            var result = [];
            if (checkList.length > 0) {
                var _loop_1 = function(checkSpec) {
                    var file = checkSpec.fileName;
                    var project = checkSpec.project;
                    if (project.getSourceFileFromName(file, reqOpen)) {
                        var diagnostics = [];
                        var syntacticDiagnostics = project.compilerService.languageService.getSyntacticDiagnostics(file);
                        if (syntacticDiagnostics && syntacticDiagnostics.length > 0) {
                            var bakedDiagnostics = syntacticDiagnostics.map(function (el) { return formatDiagnostic(file, checkSpec.project, el); });
                            diagnostics = diagnostics.concat(bakedDiagnostics);
                        }
                        var semanticDiagnostics = project.compilerService.languageService.getSemanticDiagnostics(file);
                        if (semanticDiagnostics && semanticDiagnostics.length > 0) {
                            var bakedSemanticDiagnostics = semanticDiagnostics.map(function (el) { return formatDiagnostic(file, checkSpec.project, el); });
                            diagnostics = diagnostics.concat(bakedSemanticDiagnostics);
                        }
                        if (diagnostics && diagnostics.length > 0) {
                            result.push({
                                file: file,
                                diagnostics: diagnostics
                            });
                        }
                    }
                };
                for (var _i = 0, checkList_1 = checkList; _i < checkList_1.length; _i++) {
                    var checkSpec = checkList_1[_i];
                    _loop_1(checkSpec);
                }
            }
            result = util_1.appendProjectErrors(result, projectsToProcess, this.getIDEProjectService().projectErrors, hasEmptyProject, ts_impl);
            return result;
        };
        IDESession.prototype.getProjectDiagnosticsForFileExt = function (fileName) {
            fileName = ts_impl.normalizePath(fileName);
            var project = this.projectService.getProjectForFile(fileName);
            return this.getProjectDiagnosticsExt(project);
        };
        IDESession.prototype.getMainFileDiagnosticsForFileExt = function (fileName) {
            if (mainFile == null) {
                return this.getDiagnosticsExt([fileName]);
            }
            mainFile = ts_impl.normalizePath(mainFile);
            fileName = ts_impl.normalizePath(fileName);
            var project = this.projectService.getProjectForFile(mainFile);
            if (!project) {
                return [];
            }
            var resultDiagnostics = this.getDiagnosticsExt(project.getFileNames(), project, false);
            if (project.getSourceFileFromName(fileName) == null) {
                if (resultDiagnostics == null) {
                    resultDiagnostics = [];
                }
                resultDiagnostics.push({
                    file: fileName,
                    diagnostics: [{
                            start: null,
                            end: null,
                            text: "File was not processed because there is no a reference from main file"
                        }]
                });
            }
            return resultDiagnostics;
        };
        IDESession.prototype.getProjectDiagnosticsExt = function (project) {
            if (!project) {
                return [];
            }
            var program = project.compilerService.languageService.getProgram();
            var diagnostics = [];
            var syntax = program.getSyntacticDiagnostics();
            if (syntax && syntax.length > 0) {
                diagnostics = diagnostics.concat(syntax);
            }
            var global = program.getGlobalDiagnostics();
            if (global && global.length > 0) {
                diagnostics = diagnostics.concat(global);
            }
            var semantic = program.getSemanticDiagnostics();
            if (semantic && semantic.length > 0) {
                diagnostics = diagnostics.concat(semantic);
            }
            if (ts_impl.sortAndDeduplicateDiagnostics) {
                diagnostics = ts_impl.sortAndDeduplicateDiagnostics(diagnostics);
            }
            var fileToDiagnostics = {};
            var result = [];
            for (var _i = 0, diagnostics_1 = diagnostics; _i < diagnostics_1.length; _i++) {
                var diagnostic = diagnostics_1[_i];
                var sourceFile = diagnostic.file;
                if (!sourceFile) {
                    result.push({
                        file: project.projectFilename,
                        diagnostics: [formatDiagnostic(undefined, project, diagnostic)]
                    });
                }
                var fileName = ts_impl.normalizePath(sourceFile.fileName);
                var fileDiagnostics = fileToDiagnostics[fileName];
                if (!fileDiagnostics) {
                    fileDiagnostics = [];
                    fileToDiagnostics[fileName] = fileDiagnostics;
                }
                fileDiagnostics.push(formatDiagnostic(fileName, project, diagnostic));
            }
            if (diagnostics && diagnostics.length > 0) {
                for (var fileName in fileToDiagnostics) {
                    if (fileToDiagnostics.hasOwnProperty(fileName)) {
                        var resultDiagnostic = fileToDiagnostics[fileName];
                        if (resultDiagnostic) {
                            result.push({
                                file: fileName,
                                diagnostics: resultDiagnostic
                            });
                        }
                    }
                }
            }
            var projectsToProcess = {};
            var hasEmptyProject = false;
            if (project.projectFilename) {
                projectsToProcess[project.projectFilename] = project;
            }
            else {
                hasEmptyProject = true;
            }
            result = util_1.appendProjectErrors(result, projectsToProcess, this.getIDEProjectService().projectErrors, hasEmptyProject, ts_impl);
            return result;
        };
        IDESession.prototype.ensureDirectoriesExist = function (directoryPath) {
            if (directoryPath.length > ts_impl.getRootLength(directoryPath) && !this._host.directoryExists(directoryPath)) {
                var parentDirectory = ts_impl.getDirectoryPath(directoryPath);
                this.ensureDirectoriesExist(parentDirectory);
                this._host.createDirectory(directoryPath);
            }
        };
        return IDESession;
    }(TypeScriptSession));
    var IDEProjectService = (function (_super) {
        __extends(IDEProjectService, _super);
        function IDEProjectService(host, psLogger, eventHandler) {
            _super.call(this, host, psLogger, eventHandler);
            this.projectEmittedWithAllFiles = {};
            this.projectErrors = {};
        }
        IDEProjectService.prototype.openClientFileExt = function (fileName, fileContent, configFileName) {
            if (configFileName) {
                this.openOrUpdateConfiguredProjectForFile(ts_impl.normalizePath(configFileName));
            }
            else {
                if (isVersionTypeScript15)
                    return ts15impl_1.openClientFileConfig15(this, fileName, fileContent, ts_impl);
                this.openOrUpdateConfiguredProjectForFile(fileName);
            }
            var info = this.openFile(fileName, /*openedByClient*/ true, fileContent);
            this.addOpenFile(info);
            return info;
        };
        IDEProjectService.prototype.createInferredProject = function (root) {
            var project = _super.prototype.createInferredProject.call(this, root);
            if (commonDefaultOptions && project && project.compilerService) {
                var commonSettings = project.compilerService.settings;
                var res = {};
                if (commonSettings) {
                    for (var id in commonSettings) {
                        res[id] = commonSettings[id];
                    }
                }
                for (var id in commonDefaultOptions) {
                    res[id] = commonDefaultOptions[id];
                }
                project.compilerService.setCompilerOptions(res);
            }
            return project;
        };
        IDEProjectService.prototype.cleanCachedData = function (projectFilename) {
            if (projectFilename) {
                this.projectEmittedWithAllFiles[projectFilename] = null;
                this.projectErrors[projectFilename] = null;
            }
        };
        IDEProjectService.prototype.watchedProjectConfigFileChanged = function (project) {
            var projectFilename = project.projectFilename;
            this.cleanCachedData(projectFilename);
            if (isVersionTypeScript15) {
                return;
            }
            _super.prototype.watchedProjectConfigFileChanged.call(this, project);
            logger_impl_1.serverLogger("Watcher — project changed " + (projectFilename ? projectFilename : "unnamed"), true);
        };
        IDEProjectService.prototype.configFileToProjectOptions = function (configFilename) {
            var normalizedConfigName = ts_impl.normalizePath(configFilename);
            if (normalizedConfigName) {
                if (logger_impl_1.isLogEnabled) {
                    logger_impl_1.serverLogger("Parse config normalized path " + normalizedConfigName);
                }
                this.projectErrors[normalizedConfigName] = null;
            }
            if (isVersionTypeScript15 || isVersionTypeScript16) {
                var result = _super.prototype.configFileToProjectOptions.call(this, configFilename);
                this.setProjectLevelError(result, configFilename);
                return result;
            }
            var configFileToProjectOptions = this.configFileToProjectOptionsExt(configFilename);
            configFileToProjectOptions = util_1.copyOptionsWithResolvedFilesWithoutExtensions(configFileToProjectOptions, ts_impl);
            if (logger_impl_1.isLogEnabled) {
                logger_impl_1.serverLogger("Parse config result options: " + JSON.stringify(configFileToProjectOptions));
            }
            this.setProjectLevelError(configFileToProjectOptions, normalizedConfigName);
            return configFileToProjectOptions;
        };
        IDEProjectService.prototype.setProjectLevelError = function (configFileToProjectOptions, normalizedConfigName) {
            if (configFileToProjectOptions.errors) {
                var errors = configFileToProjectOptions.errors;
                if (errors.length > 0) {
                    var errorsForService = errors.map(function (el) {
                        return {
                            end: undefined,
                            start: undefined,
                            text: ts_impl.flattenDiagnosticMessageText(el.messageText, "\n")
                        };
                    });
                    this.projectErrors[normalizedConfigName] = errorsForService;
                    //back compatibility 1.8
                    configFileToProjectOptions.error = { errorMsg: errorsForService[0].text };
                }
            }
            else if (configFileToProjectOptions.error) {
                var error = configFileToProjectOptions.error;
                var errorMessage = error.errorMsg ? error.errorMsg : "Error parsing tsconfig";
                this.projectErrors[normalizedConfigName] = [{
                        text: errorMessage,
                        end: undefined,
                        start: undefined
                    }];
            }
        };
        /**
         * copy of super#configFileToProjectOptions()
         */
        IDEProjectService.prototype.configFileToProjectOptionsExt = function (configFilename) {
            if (logger_impl_1.isLogEnabled) {
                logger_impl_1.serverLogger("Parse config " + configFilename);
            }
            configFilename = ts_impl.normalizePath(configFilename);
            // file references will be relative to dirPath (or absolute)
            var dirPath = ts_impl.getDirectoryPath(configFilename);
            var contents = this.host.readFile(configFilename);
            var rawConfig = ts_impl.parseConfigFileTextToJson(configFilename, contents);
            if (rawConfig.error) {
                if (logger_impl_1.isLogEnabled) {
                    logger_impl_1.serverLogger("Parse config error " + JSON.stringify(rawConfig.error));
                }
                return { succeeded: false, errors: [rawConfig.error] };
            }
            else {
                var parsedJsonConfig = rawConfig.config;
                var parsedCommandLine = ts_impl.parseJsonConfigFileContent(parsedJsonConfig, this.host, dirPath, /*existingOptions*/ {}, configFilename);
                if (parsedCommandLine.errors && (parsedCommandLine.errors.length > 0)) {
                    var result = { succeeded: false, errors: parsedCommandLine.errors };
                    // if (parsedCommandLine.fileNames && parsedCommandLine.options) {
                    //     const projectOptions = this.createProjectOptions(parsedCommandLine, parsedJsonConfig);
                    //     result.projectOptions = projectOptions;
                    // }
                    return result;
                }
                else {
                    var projectOptions = this.createProjectOptionsForCommandLine(parsedCommandLine, parsedJsonConfig);
                    return { succeeded: true, projectOptions: projectOptions };
                }
            }
        };
        IDEProjectService.prototype.createProjectOptionsForCommandLine = function (parsedCommandLine, parsedJsonConfig) {
            var projectOptions = {
                files: parsedCommandLine.fileNames,
                compilerOptions: parsedCommandLine.options
            };
            if (parsedCommandLine.wildcardDirectories) {
                projectOptions.wildcardDirectories = parsedCommandLine.wildcardDirectories;
            }
            if (parsedJsonConfig && parsedJsonConfig.compileOnSave === false) {
                projectOptions.compileOnSave = false;
            }
            return projectOptions;
        };
        return IDEProjectService;
    }(TypeScriptProjectService));
    /**
     * copy formatDiag method (but we use 'TS' prefix)
     */
    function formatDiagnostic(fileName, project, diagnostic) {
        return {
            start: project.compilerService.host.positionToLineOffset(fileName, diagnostic.start),
            end: project.compilerService.host.positionToLineOffset(fileName, diagnostic.start + diagnostic.length),
            text: "TS" + diagnostic.code + ":" + ts_impl.flattenDiagnosticMessageText(diagnostic.messageText, "\n")
        };
    }
    function fixNameWithProcessor(filename, onError, contentRoot, sourceRoot) {
        if (pathProcessor) {
            filename = pathProcessor.getExpandedPath(filename, contentRoot, sourceRoot, onError);
        }
        return filename;
    }
    var ideSession = new IDESession(ts_impl.sys, Buffer.byteLength, process.hrtime, logger);
    return ideSession;
}
exports.getSession = getSession;
