/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database.model;

import com.intellij.openapi.util.Conditions;
import com.intellij.openapi.util.Couple;
import com.intellij.openapi.util.Pair;
import com.intellij.util.Functions;
import com.intellij.util.containers.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

/**
 * @author gregsh
 */
public final class MetaModel<T> implements Comparator<ObjectKind> {

  @NotNull
  public static <T> MetaModelBuilder<T> builder() {
    return new MetaModelBuilder<>();
  }

  private final Map<Couple<ObjectKind>, Class> objectClasses;
  private final Map<String, ObjectKind> objectKinds;
  private final Set<ObjectKind> namespaces;
  private final MultiMap<ObjectKind, List<ObjectKind>> paths = MultiMap.createLinkedSet();

  private MetaModel(@NotNull Map<Couple<ObjectKind>, Class> objectClasses, @NotNull Map<String, ObjectKind> objectKinds) {
    this.objectClasses = objectClasses;
    this.objectKinds = objectKinds;
    namespaces = Collections.unmodifiableSet(JBIterable.from(objectClasses.keySet()).filter(
      Conditions.compose(Functions.fromMap(objectClasses), Conditions.assignableTo(DasNamespace.class))).
      transform(Functions.pairSecond()).addAllTo(ContainerUtil.newLinkedHashSet()));
    Set<ObjectKind> visited = ContainerUtil.newLinkedHashSet();
    JBIterable<ObjectKind> traversal = new JBTreeTraverser<>(this::getChildKinds)
      .withRoot(ObjectKind.ROOT)
      .traverse(TreeTraversal.INTERLEAVED_DFS);
    for (TreeTraversal.TracingIt<ObjectKind> it = traversal.typedIterator(); it.hasNext(); ) {
      ObjectKind next = it.next();
      List<ObjectKind> list = it.backtrace().toList();
      if (list.size() <= objectKinds.size()) {
        paths.putValue(next, ContainerUtil.newUnmodifiableList(list));
      }
      visited.add(next);
      if (visited.size() == objectKinds.size()) break;
    }
  }

  public Class<? extends T> getObjectClass(ObjectKind parent, ObjectKind child) {
    return objectClasses.get(Couple.of(parent, child));
  }

  public Class<? extends T> getObjectClass(Couple<ObjectKind> couple) {
    return objectClasses.get(couple);
  }

  @NotNull
  public Set<Couple<ObjectKind>> getKindCouples() {
    return objectClasses.keySet();
  }

  @NotNull
  public Iterable<ObjectKind> getKinds() {
    return objectKinds.values();
  }

  @NotNull
  public JBIterable<ObjectKind> getRootNamespaceKinds() {
    return getChildKinds(ObjectKind.ROOT);
  }

  @NotNull
  public JBIterable<ObjectKind> getChildKinds(@NotNull ObjectKind kind) {
    return JBIterable.from(getKindCouples()).
      filter(Conditions.<Pair<ObjectKind, ObjectKind>, ObjectKind>compose(Functions.<ObjectKind>pairFirst(), Conditions.is(kind))).
      transform(Functions.pairSecond());
  }

  @NotNull
  public JBIterable<ObjectKind> getParentKinds(@NotNull ObjectKind kind) {
    return JBIterable.from(getKindCouples()).
      filter(Conditions.<Pair<ObjectKind, ObjectKind>, ObjectKind>compose(Functions.<ObjectKind>pairSecond(), Conditions.is(kind))).
      transform(Functions.pairFirst());
  }

  @NotNull
  public Set<ObjectKind> getNamespaces() {
    return namespaces;
  }

  @NotNull
  public JBIterable<List<ObjectKind>> getPathsToRoot(ObjectKind kind) {
    if (!paths.containsKey(kind)) return JBIterable.empty();
    return JBIterable.from(paths.get(kind));
  }

  @Nullable
  public ObjectKind findKind(@Nullable String code) {
    if (code == null) return null;
    code = code.trim();
    if (code.isEmpty()) return null;
    return objectKinds.get(code);
  }

  @Override
  public int compare(ObjectKind o1, ObjectKind o2) {
    if (o1 == o2) return 0;
    for (ObjectKind t : objectKinds.values()) {
      if (o1 == t) return -1;
      if (o2 == t) return 1;
    }
    throw new AssertionError(o1 + " and " + o2 + " not found in " + this);
  }

  @Override
  public String toString() {
    return objectKinds.values().toString();
  }

  public static final class MetaModelBuilder<T> {
    private final Map<Couple<ObjectKind>, Class> a = ContainerUtil.newLinkedHashMap();
    private final Map<String, ObjectKind> b = ContainerUtil.newLinkedHashMap();

    public MetaModelBuilder() {
      b.put("root", ObjectKind.ROOT);
    }

    public MetaModelBuilder<T> put(ObjectKind parent, ObjectKind child, Class<? extends T> childClass) {
      a.put(Couple.of(parent, child), childClass);
      b.put(child.code(), child);
      return this;
    }

    public MetaModel<T> build() {
      return new MetaModel<>(Collections.unmodifiableMap(a), Collections.unmodifiableMap(b));
    }
  }
}
