/*
 * Copyright 2000-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.sql.formatter.settings;

import com.intellij.database.util.Case;
import com.intellij.psi.codeStyle.CodeStyleSettings;
import com.intellij.psi.codeStyle.CommonCodeStyleSettings;
import com.intellij.psi.codeStyle.CustomCodeStyleSettings;
import com.intellij.sql.util.NameTemplate;
import com.intellij.util.ArrayUtil;
import org.intellij.lang.annotations.MagicConstant;
import org.jetbrains.annotations.NotNull;

/**
 * @author Gregory.Shrago, ignatov
 */
public class SqlCodeStyleSettings extends CustomCodeStyleSettings {
  public static final int TO_UPPER = 0;
  public static final int TO_LOWER = 1;
  public static final int DO_NOT_CHANGE = 2;
  public static final int AS_KEYWORDS = 3;

  public static final int QUOTE = 0;
  public static final int UNQUOTE = 1;

  public static int[] TOKEN_CASE = {TO_UPPER, TO_LOWER, DO_NOT_CHANGE};
  public static int[] TOKEN_CASE_EXT = ArrayUtil.mergeArrays(new int[]{AS_KEYWORDS}, TOKEN_CASE);
  public static String[] TOKEN_CASE_NAMES = {"To upper", "To lower", "Do not change"};
  public static String[] TOKEN_CASE_NAMES_EXT = ArrayUtil.mergeArrays(new String []{"As keywords"}, TOKEN_CASE_NAMES);
  public static int[] QUOTE_OPTIONS = {QUOTE, UNQUOTE, DO_NOT_CHANGE};
  public static String[] QUOTE_OPTIONS_NAMES = {"Quote", "Unquote", "Do not change"};

  public boolean ALIGN_AS_IN_SELECT_STATEMENT = true;
  public boolean ALIGN_TYPE_IN_CREATE_STATEMENT = true;
  public boolean ALIGN_TYPE_IN_BLOCK_STATEMENT = true;
  public boolean ALIGN_TYPE_IN_ARGUMENT_DEFINITION = true;
  public boolean ALIGN_INSIDE_BINARY_EXPRESSION = true;
  public boolean ALIGN_INSIDE_QUERY_EXPRESSION = true;
  public boolean ALIGN_EQ_INSIDE_SET_CLAUSE = true;  

  public boolean SPACES_AROUND_OPERATORS = true;
  public boolean NEW_LINE_BEFORE_FROM = true;
  public boolean NEW_LINE_BEFORE_JOIN = true;
  public boolean NEW_LINE_BEFORE_JOIN_CONDITION = false;
  public boolean NEW_LINE_BEFORE_WHERE = true;
  public boolean NEW_LINE_BEFORE_GROUP_BY = true;
  public boolean NEW_LINE_BEFORE_ORDER_BY = true;
  public boolean NEW_LINE_BEFORE_HAVING = true;
  public boolean NEW_LINE_BEFORE_THEN = true;
  public boolean NEW_LINE_BEFORE_ELSE = true;
  public boolean NEW_LINE_BEFORE_OTHER_CLAUSES = true;
  public boolean NEW_LINE_BEFORE_COMMA = false;
  public boolean NEW_LINE_BEFORE_QUERY_INSIDE_PARENTHESIS = false;
  public boolean NEW_LINE_BEFORE_QUERY_INSIDE_DML = false;
  public boolean NEW_LINE_AROUND_SEMICOLON = false;

  public boolean INDENT_JOIN = true;
  public boolean INDENT_JOIN_CONDITION = true;
  public boolean INDENT_SELECT_INTO_CLAUSE = false;

  public int WRAP_INSIDE_CREATE_TABLE = CommonCodeStyleSettings.WRAP_ALWAYS;
  public int WRAP_INSIDE_SELECT = CommonCodeStyleSettings.WRAP_AS_NEEDED;
  public int WRAP_INSIDE_JOIN_EXPRESSION = CommonCodeStyleSettings.WRAP_AS_NEEDED;
  public int WRAP_INSIDE_GROUP_BY = CommonCodeStyleSettings.WRAP_AS_NEEDED;
  public int WRAP_INSIDE_WHERE = CommonCodeStyleSettings.WRAP_AS_NEEDED;
  public int WRAP_INSIDE_ORDER_BY = CommonCodeStyleSettings.WRAP_AS_NEEDED;
  public int WRAP_INSIDE_SET = CommonCodeStyleSettings.WRAP_AS_NEEDED;
  public int WRAP_INSIDE_BINARY_EXPRESSION = CommonCodeStyleSettings.WRAP_AS_NEEDED;
  public int WRAP_INSIDE_ARGUMENT_DEFINITION = CommonCodeStyleSettings.WRAP_AS_NEEDED;
  public int WRAP_INSIDE_CALL_EXPRESSION = CommonCodeStyleSettings.WRAP_AS_NEEDED;
  public int WRAP_INSIDE_VALUES_EXPRESSION = CommonCodeStyleSettings.WRAP_AS_NEEDED;
  public int WRAP_VALUES_EXPRESSION = CommonCodeStyleSettings.WRAP_AS_NEEDED;
  public int WRAP_PARENTHESIZED_EXPRESSION_INSIDE_VALUES = CommonCodeStyleSettings.WRAP_AS_NEEDED;

  public static final int ALWAYS = 0;
  public static final int IF_MORE_THAN_ONE = 1;
  public static final int NEVER = 2;

  public boolean DISABLE_FORMATTING = false;

  public static int[] NEW_LINE_AFTER_SELECT_VALUES = {ALWAYS, IF_MORE_THAN_ONE, NEVER};
  public static String[] NEW_LINE_AFTER_SELECT_NAMES = {"Always", "If more than one", "Never"};

  @Deprecated()
  public boolean NEW_LINE_AFTER_SELECT = true;
  
  @SuppressWarnings("deprecation")
  @MagicConstant(intValues = {ALWAYS, IF_MORE_THAN_ONE, NEVER})
  public int NEW_LINE_AFTER_SELECT_2 = !NEW_LINE_AFTER_SELECT ? NEVER : IF_MORE_THAN_ONE;  // migration

  public boolean NEW_LINE_AFTER_SELECT_ITEM = true;

  @MagicConstant(intValues = {QUOTE, UNQUOTE, DO_NOT_CHANGE})
  public int QUOTE_IDENTIFIER = DO_NOT_CHANGE;

  @MagicConstant(intValues = {TO_UPPER, TO_LOWER, DO_NOT_CHANGE})
  @interface IdentifierCase {}

  @MagicConstant(intValues = {TO_UPPER, TO_LOWER, DO_NOT_CHANGE, AS_KEYWORDS})
  @interface IdentifierCaseExt {}

  @IdentifierCase
  public int KEYWORD_CASE = TO_UPPER;
  @IdentifierCaseExt
  public int TYPE_CASE = AS_KEYWORDS;
  @IdentifierCase
  public int IDENTIFIER_CASE = DO_NOT_CHANGE;
  @IdentifierCase
  public int QUOTED_IDENTIFIER_CASE = DO_NOT_CHANGE;

  public SqlCodeStyleSettings(CodeStyleSettings container) {
    super("SqlCodeStyleSettings", container);
  }

  @NotNull
  public static Case getCaseMode(@IdentifierCase int codeStyleSetting) {
    return codeStyleSetting == TO_UPPER ? Case.UPPER :
           codeStyleSetting == TO_LOWER ? Case.LOWER :
           Case.MIXED;
  }

  @NotNull
  public Case getCaseModeExt(@IdentifierCaseExt int codeStyleSetting) {
    if (codeStyleSetting == AS_KEYWORDS) return getCaseModeExt(KEYWORD_CASE);
    return getCaseMode(codeStyleSetting);
  }

  public String INDEX_NAME_TEMPLATE = NameTemplate.IndexNameTemplate.DEFAULT ;
  public String PRIMARY_KEY_NAME_TEMPLATE = NameTemplate.PrimaryKeyNameTemplate.DEFAULT;
  public String FOREIGN_KEY_NAME_TEMPLATE = NameTemplate.ForeignKeyNameTemplate.DEFAULT;
}