---- QuerySessionCurrentInfo ----
select current_database(),
       current_schema(),
       current_timestamp as current_timestamp
;


---- ListDatabases ----
select oid::bigint as id, datname as name
from pg_catalog.pg_database
where not datistemplate
  and datname = current_database() -- temporary, until DBE-2287 is not implemented
order by case when datname = current_database() then -1::bigint else oid::bigint end
;


---- ListSchemas ----
select oid::bigint as id, nspname as name
from pg_catalog.pg_namespace
order by case when nspname = current_schema() then -1::bigint else oid::bigint end
;


---- RetrieveSchemas ----
select S.nspname as schema_name,
       S.oid as schema_id,
       S.xmin as schema_state_number,
       pg_catalog.pg_get_userbyid(S.nspowner) as owner_name,
       D.description
from pg_catalog.pg_namespace S
     left outer join pg_catalog.pg_description D on S.oid = D.objoid
order by (S.nspname = current_schema) desc,
         strpos(S.nspname,'pg_temp'), strpos(S.nspname,'pg_toast'),
         case when D.description is not null then 1 else 2 end
;


---- Now ----
select current_timestamp
;


---- CountSchemaStateNumber ----
select max(last_tx) as state_number
from (
  select max(xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_type
    where typnamespace = oid(?)
  union all
  select max(xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_class
    where relnamespace = oid(?)
  union all
  select max(xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_proc
    where pronamespace = oid(?)
  union all
  select max(xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_operator
    where oprnamespace = oid(?)
  union all
  select max(A.xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_attribute A join pg_catalog.pg_class K on K.oid = A.attrelid
    where K.relnamespace = oid(?)
  union all
  select max(xmin::varchar::bigint) as last_tx
    from pg_catalog.pg_constraint
    where connamespace = oid(?)
) X
;


---- ListExistentClasses ----
select distinct oid
from pg_catalog.pg_type
where typnamespace = oid(?)
union
select distinct oid
from pg_catalog.pg_class
where relnamespace = oid(?)
union
select distinct oid
from pg_catalog.pg_proc
where pronamespace = oid(?)
union
select distinct oid
from pg_catalog.pg_operator
where oprnamespace = oid(?)
union
select distinct oid
from pg_catalog.pg_constraint
where connamespace = oid(?)
;


---- RetrieveSequences90 ----
select oid as sequence_id,
       xmin as sequence_state_number,
       relname as sequence_name,
       null as sequence_params
from pg_catalog.pg_class C
where relnamespace = oid(?)
  and relkind = 'S'
  and xmin::varchar::bigint > ?
;

---- RetrieveSequences91 ----
with U as ( select usesysid as current_user_id
            from pg_catalog.pg_user
            where usename = current_user )
select oid as sequence_id,
       xmin as sequence_state_number,
       relname as sequence_name,
       case when relowner = current_user_id
            then pg_catalog.pg_sequence_parameters(C.oid)::varchar
            else null
            end as sequence_params
from pg_catalog.pg_class C,
     U
where relnamespace = oid(?)
  and relkind = 'S'
  and xmin::varchar::bigint > ?
;


---- RetrieveDataTypes ----
select T.oid as type_id,
       T.xmin as type_state_number,
       T.typname as type_name,
       T.typtype as type_sub_kind,
       T.typcategory as type_category,
       T.typrelid as class_id,
       T.typbasetype as base_type_id,
       case when T.typtype in ('c','e') then null
            else pg_catalog.format_type(T.typbasetype, T.typtypmod) end as type_def
from pg_catalog.pg_type T
         left outer join pg_catalog.pg_class C
             on T.typrelid = C.oid
where T.typnamespace = oid(?)
  and T.xmin::varchar::bigint > ?
  and (T.typtype in ('d','e') or C.relkind = 'c'::"char" or T.typtype = 'b' and T.typelem = 0)
order by 1
;


---- RetrieveTables ----
select T.relkind as table_kind,
       T.relname as table_name,
       T.oid as table_id,
       T.xmin as table_state_number,
       T.relhasoids as table_with_oids,
       T.relnamespace as tablespace_id,
       pg_get_userbyid(relowner) as owner_name
from pg_catalog.pg_class T
where relnamespace = oid(?)
  and relkind in ('r', 'm', 'v')
  and T.xmin::varchar::bigint > ?
order by table_kind, table_id
;


---- RetrieveStructAttributes ----
with S as ( select T.oid as type_id, C.oid as class_id
            from pg_catalog.pg_type T,
                 pg_catalog.pg_class C
            where T.typnamespace = oid(?)
              and T.typnamespace = C.relnamespace
              and T.xmin::varchar::bigint > ?
              and T.typrelid = C.oid
              and T.typtype = 'c'::"char"
              and relkind = 'c'::"char" ),
     A as ( select attrelid as class_id,
                   attnum as att_position,
                   attname as att_name,
                   pg_catalog.format_type(atttypid,atttypmod) as att_type
            from pg_catalog.pg_attribute )
select type_id, att_position, att_name, att_type
from S natural join A
order by type_id::bigint, att_position
;


---- RetrieveEnumLabels90 ----
select T.oid as enum_id,
       array_agg(E.enumlabel::varchar) as enum_labels
from pg_catalog.pg_type T,
     pg_catalog.pg_enum E
where T.typnamespace = oid(?)
  and T.xmin::varchar::bigint > ?
  and T.oid = E.enumtypid
group by T.oid
;

---- RetrieveEnumLabels91 ----
select T.oid as enum_id,
       array_agg(E.enumlabel::varchar order by E.enumsortorder) as enum_labels
from pg_catalog.pg_type T,
     pg_catalog.pg_enum E
where T.typnamespace = oid(?)
  and T.xmin::varchar::bigint > ?
  and T.oid = E.enumtypid
group by T.oid
;


---- RetrieveColumns ----
with T as ( select distinct T.oid as table_id
            from pg_catalog.pg_class T,
                 pg_catalog.pg_attribute A
            where T.relnamespace = oid(?)
              and T.relkind in ('r', 'm', 'v')
              and T.oid = A.attrelid
              and A.xmin::varchar::bigint > ? )
select T.table_id,
       C.attnum as column_position,
       C.attname as column_name,
       C.xmin as column_state_number,
       C.atttypmod as type_mod,
       C.attndims as dimensions_number,
       pg_catalog.format_type(C.atttypid, C.atttypmod) as type_spec,
       C.attnotnull as mandatory,
       D.adsrc as column_default_expression,
       C.attisdropped as column_is_dropped
from T
  inner join pg_catalog.pg_attribute C on T.table_id = C.attrelid
  left outer join pg_catalog.pg_attrdef D on (C.attrelid, C.attnum) = (D.adrelid, D.adnum)
where attnum > 0
  and not attisdropped
order by table_id, attnum
;


---- RetrieveIndices ----
with tab as ( select relnamespace as schema_id,
                     oid as table_id,
                     relkind as table_kind
              from pg_catalog.pg_class
              where relnamespace = oid(?)
                and relkind in ('r','m','v')),
    ind_head as ( select indrelid as table_id,
                         indexrelid as index_id,
                         indisunique as is_unique,
                         indisprimary as is_primary,
                         indkey as column_positions,
                         indclass as column_operators,
                         indoption as column_options,
                         indpred as predicate
                  from pg_catalog.pg_index ),
    ind_stor as ( select relnamespace as schema_id,
                         oid as index_id,
                         relname as index_name
                  from pg_catalog.pg_class
                  where relkind = 'i'
                    and xmin::varchar::bigint > ?)
select table_id, table_kind,
       index_name, index_id,
       is_unique, is_primary,
       column_positions, column_operators, column_options,
       predicate
from tab natural join ind_head natural join ind_stor
;


---- RetrieveConstraints ----
with I as ( select ?::oid as schema_id,
                   ?::bigint as state_id ),
     T as ( select relnamespace as schema_id,
                   oid as table_id,
                   relkind as table_kind,
                   xmin::varchar::bigint as table_state_id
            from pg_catalog.pg_class
            where relkind in ('r','v') ),
     C as ( select connamespace as schema_id,
                   conrelid as table_id,
                   oid::bigint as con_id,
                   xmin::varchar::bigint as con_state_id,
                   conname as con_name,
                   contype as con_kind,
                   conkey as con_columns,
                   conindid as index_id,
                   confrelid as ref_table_id,
                   confkey as ref_columns,
                   consrc as con_expression
            from pg_catalog.pg_constraint
            where contype in ('p','u','f','c') )
select table_id,
       table_kind,
       con_id,
       con_name,
       con_kind,
       con_columns,
       index_id,
       ref_table_id,
       con_expression
from I natural join T natural join C
where table_state_id > state_id
   or con_state_id > state_id
order by strpos('pufc', con_kind)
;


---- RetrieveUsedArgumentTypes ----
with schema_procs as (select prorettype, proargtypes, proallargtypes
                      from pg_catalog.pg_proc
                      where pronamespace = oid(?)
                        and xmin::varchar::bigint > ? ),
     schema_arg_types as ( select prorettype as type_id
                           from schema_procs
                           union
                           select distinct unnest(proargtypes) as type_id
                           from schema_procs
                           union
                           select distinct unnest(proallargtypes) as type_id
                           from schema_procs )
select type_id, pg_catalog.format_type(type_id, null) as type_spec
from schema_arg_types
;


---- RetrieveRoutines ----
with languages as (select oid as lang_oid, lanname as lang
                   from pg_catalog.pg_language),
     routines as (select proname as r_name,
                         prolang as lang_oid,
                         oid as r_id,
                         xmin as r_state_number,
                         proargnames as arg_names,
                         proargmodes as arg_modes,
                         proargtypes::int[] as in_arg_types,
                         proallargtypes::int[] as all_arg_types,
                         pg_catalog.pg_get_expr(proargdefaults, 0) as arg_defaults,
                         provariadic as arg_variadic_id,
                         prorettype as ret_type_id,
                         proretset as ret_set,
                         proisagg as is_aggregate,
                         proiswindow as is_window,
                         provolatile as volatile_kind
                  from pg_catalog.pg_proc
                  where pronamespace = oid(?)
                    and xmin::varchar::bigint > ?)
select *
from routines natural join languages
;


---- RetrieveOperators ----
select O.oid as op_id,
       oprname as op_name,
       oprkind as op_kind,
       oprleft as arg_left_type_id,
       oprright as arg_right_type_id,
       oprresult as arg_result_type_id,
       oprcode as func_main
from pg_catalog.pg_operator O
where oprnamespace = oid(?)
  and xmin::varchar::bigint > ?
;


---- RetrieveRules ----
select R.ev_class                       as table_id,
       R.oid                            as rule_id,
       R.xmin                           as rule_state_number,
       R.rulename                       as rule_name,
       translate(ev_type,'1234','SIUD') as rule_event_code,
       R.ev_enabled                     as rule_fire_mode,
       R.is_instead                     as rule_is_instead
from pg_catalog.pg_rewrite R
where R.ev_class in (select oid
                     from pg_catalog.pg_class
                     where relnamespace = oid(?))
  and R.xmin::varchar::bigint > ?
  and R.rulename != '_RETURN'::name
order by R.ev_class::bigint, ev_type
;


---- RetrieveDescriptions ----
with T as ( select distinct T.oid as objoid
            from pg_catalog.pg_class T,
                 pg_catalog.pg_attribute A
            where T.relnamespace = oid(?)
              and T.relkind in ('r', 'm', 'v')
              and T.oid = A.attrelid
              and A.xmin::varchar::bigint > ? )
select objoid as table_id,
       objsubid as column_position,
       description
from T natural join pg_catalog.pg_description
order by table_id, column_position
;


---- RetrieveViewSources ----
select T.relname as view_name,
       pg_catalog.pg_get_viewdef(T.oid) as source_text
from pg_catalog.pg_class T,
     pg_catalog.pg_namespace N
where N.oid = oid(?)
  and T.relnamespace = N.oid
  and T.relkind = 'v'
  and T.xmin::varchar::bigint > ?
order by T.xmin::varchar::bigint
;



