import {serverLogger, isLogEnabled} from "./logger-impl";
import {CompileInfoHolder} from "./compile-info-holder";
import {PathProcessor} from "./out-path-process";
import {onMessage15, reload15, close15, openClientFileConfig15, setGetFileNames, openProjectByConfig} from "./ts15impl";
import {
    initCommandNames,
    isTypeScript15,
    isTypeScript16,
    copyOptionsWithResolvedFilesWithoutExtensions, appendProjectErrors, isFunctionKind
} from "./util";
import CompletionsRequest = ts.server.protocol.CompletionsRequest;
import CompletionEntry = ts.CompletionEntry;
import sys = ts.sys;
import CompilerOptions = ts.CompilerOptions;


/**
 * Default tsserver implementation doesn't return response in most cases ("open", "close", etc.)
 * we want to override the behaviour and send empty-response holder
 */
const doneRequest: Response = {
    responseRequired: true,
    response: "done"
}

const DETAILED_COMPLETION_COUNT: number = 5;

type Response = {response?: any; responseRequired?: boolean};
export type ProjectResult = {succeeded: boolean; projectOptions?: ts.server.ProjectOptions; error?: ts.server.ProjectOpenResult, errors?};
export function getSession(ts_impl: any /*must be typeof ts */,
                           logger: ts.server.Logger,
                           commonDefaultOptions?: ts.CompilerOptions,
                           pathProcessor?: PathProcessor,
                           mainFile?: string): ts.server.Session {

    let TypeScriptSession: typeof ts.server.Session = ts_impl.server.Session;
    let TypeScriptProjectService: typeof ts.server.ProjectService = ts_impl.server.ProjectService;
    let TypeScriptCommandNames: typeof ts.server.CommandNames = ts_impl.server.CommandNames;
    initCommandNames(TypeScriptCommandNames);

    let isVersionTypeScript15 = isTypeScript15(ts_impl);
    let isVersionTypeScript16 = isTypeScript16(ts_impl);

    if (isVersionTypeScript15) { setGetFileNames(ts_impl.server.Project);}

    // var 
    class IDESession extends TypeScriptSession {
        private _host: ts.server.ServerHost;

        private _mySeq: number;
        private _wasFirstMessage: boolean;


        constructor(host: ts.server.ServerHost,
                    byteLength: (buf: string, encoding?: string) => number,
                    hrtime: (start?: number[]) => number[],
                    logger: ts.server.Logger) {
            if (isVersionTypeScript15) {
                super(host, <any>logger, undefined, undefined);
            } else {
                super(host, byteLength, hrtime, logger);
            }
            this._host = host;
            this._wasFirstMessage = false;

            let handler = this.projectService.eventHandler;
            //reuse handler
            this.projectService = new IDEProjectService(host, logger, handler);
        }


        send(msg: ts.server.protocol.Message) {
            const json = JSON.stringify(msg);
            this._host.write(json + "\n");
        }


        onMessage(message: string): void {
            if (isVersionTypeScript15) {onMessage15(this, message); return; }
            super.onMessage(message);
        }

        executeCommand(request: ts.server.protocol.Request): Response {
            let startTime = Date.now();
            let command = request.command;
            try {
                if (TypeScriptCommandNames.Open == command) {
                    //use own implementation
                    const openArgs = <ts.server.protocol.OpenRequestArgs>request.arguments;
                    this.openClientFileExt(openArgs);
                    return doneRequest;

                } else if (TypeScriptCommandNames.ReloadProjects == command) {
                    this.getIDEProjectService().projectEmittedWithAllFiles = {};

                    if (isVersionTypeScript15) return reload15(this, ts_impl);

                    this.updateProjectStructureExt();
                    return doneRequest;
                } else if (TypeScriptCommandNames.IDEChangeFiles == command) {
                    const updateFilesArgs: ts.server.protocol.IDEUpdateFilesContentArgs = <ts.server.protocol.IDEUpdateFilesContentArgs>request.arguments;
                    return this.updateFilesExt(updateFilesArgs);
                } else if (TypeScriptCommandNames.IDECompile == command) {
                    const fileArgs = <ts.server.protocol.IDECompileFileRequestArgs>request.arguments;
                    return this.compileFileExt(fileArgs);
                } else if (TypeScriptCommandNames.Close == command) {

                    if (isVersionTypeScript15) { close15(this, request); return doneRequest; }

                    super.executeCommand(request);
                    return doneRequest;
                } else if (TypeScriptCommandNames.IDEGetErrors == command) {
                    let args = request.arguments;
                    return {response: {infos: this.getDiagnosticsExt(args.files)}, responseRequired: true}
                } else if (TypeScriptCommandNames.IDEGetMainFileErrors == command) {
                    let args: ts.server.protocol.FileRequestArgs = request.arguments;
                    return {response: {infos: this.getMainFileDiagnosticsForFileExt(args.file)}, responseRequired: true}
                } else if (TypeScriptCommandNames.IDEGetProjectErrors == command) {
                    let args: ts.server.protocol.FileRequestArgs = request.arguments;
                    return {response: {infos: this.getProjectDiagnosticsForFileExt(args.file)}, responseRequired: true}
                } else if (TypeScriptCommandNames.IDECompletions == command) {
                    if (isVersionTypeScript15) return {response: [], responseRequired: true};

                    return this.getCompletionExt(request);
                }

                return super.executeCommand(request);
            } finally {
                if (!this._wasFirstMessage) {
                    serverLogger("TypeScript service version: " + ts_impl.version);
                    this._wasFirstMessage = true;
                }

                let processingTime = Date.now() - startTime;

                serverLogger("Message " + request.seq + " '" + command + "' server time, mills: " + processingTime, true);
            }
        }

        updateFilesExt(args: ts.server.protocol.IDEUpdateFilesContentArgs): Response {
            let updated = false;
            let files = args.files;
            for (let fileName in files) {
                if (files.hasOwnProperty(fileName)) {
                    let content = files[fileName];
                    if (content !== undefined) {
                        this.changeFileExt(fileName, content);
                        updated = true;
                    }
                }
            }

            if (args.filesToReloadContentFromDisk) {
                for (let fileName of args.filesToReloadContentFromDisk) {
                    if (!fileName) {
                        continue;
                    }

                    let file = ts_impl.normalizePath(fileName);
                    this.projectService.closeClientFile(file);

                    if (isLogEnabled) {
                        serverLogger("Update file from disk (by 'filesToReloadContentFromDisk') " + file);
                    }
                    updated = true;
                }
            }

            if (updated) {
                this.updateProjectStructureExt();
            }

            return doneRequest;
        }

        private updateProjectStructureExt() {
            let mySeq = this.getChangeSeq();
            let matchSeq = (n) => n === mySeq;
            setTimeout(() => {
                if (matchSeq(this.getChangeSeq())) {
                    let startTime = Date.now();
                    this.projectService.updateProjectStructure();
                    serverLogger("Update project structure scheduler time, mills: " + (Date.now() - startTime), true);
                }
            }, 1500);
        }

        private getChangeSeq() {
            let anyThis: any = this;
            let superClassSeq = anyThis.changeSeq;
            if (typeof superClassSeq !== "undefined") {
                return superClassSeq;
            }
            serverLogger("WARN: Used own sequence implementation (can be slow)", true);
            return this._mySeq;
        }

        openClientFileExt(openArgs: ts.server.protocol.OpenRequestArgs): {response?: any; responseRequired?: boolean} {
            const fileName = openArgs.file;
            const fileContent = openArgs.fileContent;
            const configFile = openArgs.tsConfig;
            const file = ts_impl.normalizePath(fileName);

            return (<IDEProjectService>this.projectService).openClientFileExt(file, fileContent, configFile);
        }

        changeFileExt(fileName: string, content: string, tsconfig?: string) {
            const file = ts_impl.normalizePath(fileName);
            const project = this.projectService.getProjectForFile(file);
            if (project) {
                const compilerService = project.compilerService;
                let scriptInfo = compilerService.host.getScriptInfo(file);
                if (scriptInfo != null) {
                    scriptInfo.svc.reload(content);
                    if (isLogEnabled) {
                        serverLogger("Update file reload content from text " + file);
                    }
                } else {
                    if (isLogEnabled) {
                        serverLogger("Update file scriptInfo is null " + file);
                    }
                }
            } else {
                if (isLogEnabled) {
                    serverLogger("Update file cannot find project for " + file);
                }
                this.openClientFileExt({
                    file: fileName,
                    fileContent: content,
                    tsConfig: tsconfig
                });
            }
        }

        /**
         * todo check that output files exists
         */
        compileFileExt(req: ts.server.protocol.IDECompileFileRequestArgs): Response {
            let startCompile = Date.now();

            let compileExactFile: boolean = req.file != null;
            if (!compileExactFile && !req.tsConfig) {
                return doneRequest;
            }

            let requestedFile: string = ts_impl.normalizePath(req.file ? req.file : req.tsConfig);
            if (!requestedFile) {
                return doneRequest;
            }
            let project: ts.server.Project = this.getProjectForCompileRequest(req, requestedFile);

            if (isLogEnabled) {
                serverLogger("Compile get project end time: " + (Date.now() - startCompile));
            }
            const outFiles = [];
            let includeErrors = req.includeErrors;
            let diagnostics: ts.server.protocol.DiagnosticEventBody[] = includeErrors ? [] : undefined;
            let needCompile = (req.force || !project || !project.projectOptions || project.projectOptions.compileOnSave || project.projectOptions.compileOnSave === undefined);
            if (project && needCompile) {
                let projectFilename = project.projectFilename;
                let languageService = project.compilerService.languageService;
                let program = languageService.getProgram();
                if (isLogEnabled) {
                    serverLogger("Compile get source files end time: " + program.getSourceFiles().length + "(count): time, mills: " + (Date.now() - startCompile));
                    serverLogger("Compile project Filename: " + (projectFilename ? projectFilename : "no filename"));
                }
                let options = project.projectOptions ? project.projectOptions.compilerOptions : null;
                (<any>project.compilerService.host).getNewLine = function () {
                    return ts_impl.getNewLineCharacter(options ? options : {});
                };

                let compileInfoHolder: CompileInfoHolder = null;
                if (projectFilename) {
                    compileInfoHolder = this.getIDEProjectService().projectEmittedWithAllFiles[projectFilename];
                    compileExactFile = compileExactFile && compileInfoHolder == null;
                    if (!compileInfoHolder) {
                        compileInfoHolder = new CompileInfoHolder(ts_impl);
                        this.getIDEProjectService().projectEmittedWithAllFiles[projectFilename] = compileInfoHolder;
                    }
                } else {
                    compileExactFile = false;
                }

                let fileWriteCallback = this.getFileWrite(project, outFiles, req.contentRootForMacro, req.sourceRootForMacro);
                if (!compileExactFile) {
                    serverLogger("Compile all files using cache checking", true);

                    let toUpdateFiles: ts.SourceFile[] = [];
                    let rawSourceFiles = program.getSourceFiles();
                    rawSourceFiles.forEach((val) => {
                        if (!compileInfoHolder || compileInfoHolder.checkUpdateAndAddToCache(val)) {
                            toUpdateFiles.push(val);
                        }
                    });


                    let compilerOptions = program.getCompilerOptions();
                    let useOutFile = compilerOptions && (compilerOptions.outFile || compilerOptions.out);
                    if (toUpdateFiles.length > 0) {
                        if (toUpdateFiles.length == rawSourceFiles.length || useOutFile) {
                            let emitResult: ts.EmitResult = program.emit(undefined, fileWriteCallback);
                            diagnostics = this.appendEmitDiagnostics(project, emitResult, diagnostics);
                        } else {
                            toUpdateFiles.forEach((el) => {
                                let emitResult: ts.EmitResult = program.emit(el, fileWriteCallback);
                                diagnostics = this.appendEmitDiagnostics(project, emitResult, diagnostics);
                            })
                        }
                    }


                    serverLogger("Compile end emit files: " + (Date.now() - startCompile));
                } else {
                    let sourceFile: ts.SourceFile = program.getSourceFile(requestedFile);
                    if (sourceFile) {
                        if (!compileInfoHolder || compileInfoHolder.checkUpdateAndAddToCache(sourceFile)) {
                            let emitResult: ts.EmitResult = project.program.emit(sourceFile, fileWriteCallback);
                            diagnostics = this.appendEmitDiagnostics(project, emitResult, diagnostics);

                        }
                    } else {
                        serverLogger("Compile can't find source file: shouldn't be happened")
                    }
                }
            } else {
                if (project) {
                    serverLogger("Compile skip: compileOnSave = false", true);
                } else {
                    serverLogger("Compile can't find project: shouldn't be happened", true);
                }
            }

            if (includeErrors) {
                diagnostics = diagnostics.concat(compileExactFile ?
                    this.getDiagnosticsExt([requestedFile], project) :
                    this.getProjectDiagnosticsExt(project)
                );
            }

            if (isLogEnabled) {
                serverLogger("Compile end get diagnostics time, mills: " + (Date.now() - startCompile));
            }

            return {response: {generatedFiles: outFiles, infos: diagnostics}, responseRequired: true};
        }

        private getProjectForCompileRequest(req: ts.server.protocol.IDECompileFileRequestArgs, normalizedRequestedFile: string) {
            let project: ts.server.Project = null;
            if (req.file) {
                project = this.projectService.getProjectForFile(normalizedRequestedFile);
            } else {
                if (isVersionTypeScript15)  return openProjectByConfig(this.projectService, normalizedRequestedFile, ts_impl);

                project = this.projectService.findConfiguredProjectByConfigFile(normalizedRequestedFile);
                if (!project) {
                    const configResult = this.projectService.openConfigFile(normalizedRequestedFile);
                    if (configResult && configResult.project) {
                        return configResult.project;
                    }
                }

            }
            return project;
        }

        private getIDEProjectService() {
            return (<IDEProjectService>this.projectService);
        }

        private getTime() {
            return Date.now();
        }


        appendEmitDiagnostics(project: ts.server.Project, emitResult: ts.EmitResult, diagnostics: ts.server.protocol.DiagnosticEventBody[]) {
            if (diagnostics !== undefined && emitResult && emitResult.diagnostics) {
                let emitDiagnostics = emitResult.diagnostics;
                return diagnostics.concat(emitDiagnostics.map((el) => {
                        return {file: el.file.fileName, diagnostics: [formatDiagnostic(el.file.fileName, project, el)]}
                    }
                ));
            }

            return diagnostics;
        }

        getFileWrite(project, outFiles: string[], contentRoot?, sourceRoot?) {
            return (fileName, data?, writeByteOrderMark?, onError?, sourceFiles?) => {
                let normalizedName = ts_impl.normalizePath(fileName);
                normalizedName = fixNameWithProcessor(normalizedName, onError, contentRoot, sourceRoot);
                this.ensureDirectoriesExist(ts_impl.getDirectoryPath(normalizedName));
                if (isLogEnabled) {
                    serverLogger("Compile write file: " + normalizedName);
                }
                (<any>this._host).writeFile(normalizedName, data, writeByteOrderMark, onError, sourceFiles);
                outFiles.push(normalizedName);
            }
        }

        logError(err: Error, cmd: string) {
            const typedErr = <any>err;
            serverLogger("Error processing message: " + err.message + " " + typedErr.stack, true);

            super.logError(err, cmd);
        }

        getCompletionExt(request: ts.server.protocol.Request) {
            let startDate = -1;
            if (isLogEnabled) {
                startDate = Date.now();
            }

            const args = <ts.server.protocol.CompletionsRequestArgs>request.arguments;

            let result: Response = super.executeCommand({
                command: TypeScriptCommandNames.Completions,
                arguments: args,
                seq: request.seq,
                type: request.type
            });


            if (isLogEnabled) {
                serverLogger("Completion service implementation time, mills: " + (Date.now() - startDate));
            }

            let response: ts.server.protocol.CompletionEntry[] = result.response;

            let ideCompletions = this.getDetailedIDECompletions(args, response);

            if (isLogEnabled) {
                serverLogger("Completion with detailed items time, mills: " + (Date.now() - startDate));
            }

            return {
                response: ideCompletions,
                responseRequired: true
            }
        }

        getDetailedIDECompletions(req: ts.server.protocol.CompletionsRequestArgs,
                                  entries: ts.server.protocol.CompletionEntry[]) {
            if (!entries) {
                return entries;
            }
            const file = ts_impl.normalizePath(req.file);
            const project = this.projectService.getProjectForFile(file);
            if (!project) {
                serverLogger("Can't find project: shouldn't be happened", true)
                return entries;
            }
            const compilerService = project.compilerService;
            const position = compilerService.host.lineOffsetToPosition(file, req.line, req.offset);
            let count = 0;
            return entries.reduce((accum: ts.server.protocol.CompletionEntry[], entry: CompletionEntry) => {

                if (!isFunctionKind(entry.kind) || count++ > DETAILED_COMPLETION_COUNT) {
                    accum.push(entry);
                } else {
                    const details: any = compilerService.languageService.getCompletionEntryDetails(file, position, entry.name);
                    if (details) {
                        details.sortText = entry.sortText;
                        accum.push(details);
                    }
                }
                return accum;
            }, []);
        }

        /**
         * Possible we can remove the implementation if we will use 'pull' events
         * now just for test we use 'blocking' implementation
         * to check speed of processing
         * todo use 'pull' implementation
         */
        private getDiagnosticsExt(fileNames: string[], commonProject?: ts.server.Project, reqOpen: boolean = true): ts.server.protocol.DiagnosticEventBody[] {
            let projectsToProcess: {[p: string]: ts.server.Project} = {};
            let hasEmptyProject = false;
            if (commonProject) {
                if (commonProject.projectFilename) {
                    projectsToProcess[commonProject.projectFilename] = commonProject;
                } else {
                    hasEmptyProject = true;
                }
            }


            const checkList = fileNames.reduce((accumulator: ts.server.PendingErrorCheck[], fileName: string) => {
                fileName = ts_impl.normalizePath(fileName);
                if (commonProject) {
                    accumulator.push({fileName, project: commonProject});
                } else {
                    const project: ts.server.Project = this.projectService.getProjectForFile(fileName);
                    if (project) {
                        accumulator.push({fileName, project});
                        if (project.projectFilename) {
                            projectsToProcess[project.projectFilename] = project;
                        } else {
                            hasEmptyProject = true;
                        }
                    }
                }
                return accumulator;
            }, []);

            let result: ts.server.protocol.DiagnosticEventBody[] = [];

            if (checkList.length > 0) {
                for (let checkSpec of checkList) {
                    let file: string = checkSpec.fileName;
                    let project = checkSpec.project;

                    if (project.getSourceFileFromName(file, reqOpen)) {
                        let diagnostics: ts.server.protocol.Diagnostic[] = [];
                        const syntacticDiagnostics: ts.Diagnostic[] = project.compilerService.languageService.getSyntacticDiagnostics(file);
                        if (syntacticDiagnostics && syntacticDiagnostics.length > 0) {
                            const bakedDiagnostics = syntacticDiagnostics.map((el: ts.Diagnostic) => formatDiagnostic(file, checkSpec.project, el));
                            diagnostics = diagnostics.concat(bakedDiagnostics);
                        }

                        const semanticDiagnostics = project.compilerService.languageService.getSemanticDiagnostics(file);
                        if (semanticDiagnostics && semanticDiagnostics.length > 0) {
                            const bakedSemanticDiagnostics = semanticDiagnostics.map((el: ts.Diagnostic) => formatDiagnostic(file, checkSpec.project, el));
                            diagnostics = diagnostics.concat(bakedSemanticDiagnostics);

                        }

                        if (diagnostics && diagnostics.length > 0) {
                            result.push({
                                file,
                                diagnostics
                            })
                        }
                    }
                }
            }

            result = appendProjectErrors(result, projectsToProcess, this.getIDEProjectService().projectErrors, hasEmptyProject, ts_impl);

            return result;
        }

        getProjectDiagnosticsForFileExt(fileName: string): ts.server.protocol.DiagnosticEventBody[] {
            fileName = ts_impl.normalizePath(fileName);

            let project = this.projectService.getProjectForFile(fileName);

            return this.getProjectDiagnosticsExt(project);
        }

        getMainFileDiagnosticsForFileExt(fileName: string): ts.server.protocol.DiagnosticEventBody[] {
            if (mainFile == null) {
                return this.getDiagnosticsExt([fileName]);
            }

            mainFile = ts_impl.normalizePath(mainFile);
            fileName = ts_impl.normalizePath(fileName);
            let project = this.projectService.getProjectForFile(mainFile);
            if (!project) {
                return [];
            }
            let resultDiagnostics = this.getDiagnosticsExt(project.getFileNames(), project, false);

            if (project.getSourceFileFromName(fileName) == null) {
                if (resultDiagnostics == null) {
                    resultDiagnostics = [];
                }
                resultDiagnostics.push({
                    file: fileName,
                    diagnostics: [{
                        start: null,
                        end: null,
                        text: "File was not processed because there is no a reference from main file"
                    }]
                });
            }

            return resultDiagnostics;
        }

        getProjectDiagnosticsExt(project: ts.server.Project): ts.server.protocol.DiagnosticEventBody[] {
            if (!project) {
                return []
            }

            let program = project.compilerService.languageService.getProgram();
            let diagnostics: ts.Diagnostic[] = [];
            let syntax = program.getSyntacticDiagnostics();
            if (syntax && syntax.length > 0) {
                diagnostics = diagnostics.concat(syntax);
            }

            let global = program.getGlobalDiagnostics();
            if (global && global.length > 0) {
                diagnostics = diagnostics.concat(global);
            }
            let semantic = program.getSemanticDiagnostics();
            if (semantic && semantic.length > 0) {
                diagnostics = diagnostics.concat(semantic);
            }

            if (ts_impl.sortAndDeduplicateDiagnostics) {
                diagnostics = ts_impl.sortAndDeduplicateDiagnostics(diagnostics);
            }

            let fileToDiagnostics: {[p:string]:ts.server.protocol.Diagnostic[]} = {};
            let result:ts.server.protocol.DiagnosticEventBody[] = [];
            for (let diagnostic of diagnostics) {
                let sourceFile = diagnostic.file;
                if (!sourceFile) {
                    result.push({
                        file:project.projectFilename,
                        diagnostics:[formatDiagnostic(undefined, project, diagnostic)]
                    })
                }

                let fileName = ts_impl.normalizePath(sourceFile.fileName);
                let fileDiagnostics = fileToDiagnostics[fileName];
                if (!fileDiagnostics) {
                    fileDiagnostics = [];
                    fileToDiagnostics[fileName] = fileDiagnostics;
                }
                fileDiagnostics.push(formatDiagnostic(fileName, project, diagnostic));
            }

            if (diagnostics && diagnostics.length > 0) {
                for (let fileName in fileToDiagnostics) {
                    if (fileToDiagnostics.hasOwnProperty(fileName)) {
                        let resultDiagnostic = fileToDiagnostics[fileName];
                        if (resultDiagnostic) {
                            result.push({
                                file: fileName,
                                diagnostics: resultDiagnostic
                            });
                        }
                    }
                }
            }

            let projectsToProcess: {[p: string]: ts.server.Project } = {};
            let hasEmptyProject = false;
            if (project.projectFilename) {
                projectsToProcess[project.projectFilename] = project;
            } else {
                hasEmptyProject = true;
            }

            result = appendProjectErrors(result, projectsToProcess, this.getIDEProjectService().projectErrors, hasEmptyProject, ts_impl);


            return result;
        }

        ensureDirectoriesExist(directoryPath: string) {
            if (directoryPath.length > ts_impl.getRootLength(directoryPath) && !this._host.directoryExists(directoryPath)) {
                const parentDirectory = ts_impl.getDirectoryPath(directoryPath);
                this.ensureDirectoriesExist(parentDirectory);
                this._host.createDirectory(directoryPath);
            }
        }
    }

    class IDEProjectService extends TypeScriptProjectService {

        projectEmittedWithAllFiles: {
            [p: string]: CompileInfoHolder
        } = {}

        projectErrors: {
            [p: string]: ts.server.protocol.Diagnostic[]
        } = {}

        constructor(host: ts.server.ServerHost,
                    psLogger: ts.server.Logger,
                    eventHandler: ts.server.ProjectServiceEventHandler) {
            super(host, psLogger, eventHandler);
        }

        openClientFileExt(fileName: string, fileContent: string, configFileName: string) {
            if (configFileName) {
                this.openOrUpdateConfiguredProjectForFile(ts_impl.normalizePath(configFileName));
            } else {

                if (isVersionTypeScript15)  return openClientFileConfig15(this, fileName, fileContent, ts_impl);

                this.openOrUpdateConfiguredProjectForFile(fileName);
            }
            const info = this.openFile(fileName, /*openedByClient*/ true, fileContent);
            this.addOpenFile(info);
            return info;
        }

        createInferredProject(root: ts.server.ScriptInfo): ts.server.Project {
            let project = super.createInferredProject(root);
            if (commonDefaultOptions && project && project.compilerService) {
                let commonSettings = project.compilerService.settings;
                let res: ts.CompilerOptions = {}
                if (commonSettings) {
                    for (const id in commonSettings) {
                        res[id] = (<any>commonSettings)[id];
                    }
                }
                for (const id in commonDefaultOptions) {
                    res[id] = (<any>commonDefaultOptions)[id];
                }

                project.compilerService.setCompilerOptions(res);
            }

            return project;
        }

        cleanCachedData(projectFilename: string) {
            if (projectFilename) {
                this.projectEmittedWithAllFiles[projectFilename] = null;
                this.projectErrors[projectFilename] = null;
            }
        }

        watchedProjectConfigFileChanged(project: ts.server.Project): void {
            let projectFilename = project.projectFilename;
            this.cleanCachedData(projectFilename);

            if (isVersionTypeScript15) { return; }

            super.watchedProjectConfigFileChanged(project);

            serverLogger("Watcher — project changed " + (projectFilename ? projectFilename : "unnamed"), true);
        }

        configFileToProjectOptions(configFilename: string): ProjectResult {

            let normalizedConfigName = ts_impl.normalizePath(configFilename);
            if (normalizedConfigName) {
                if (isLogEnabled) {
                    serverLogger("Parse config normalized path " + normalizedConfigName);
                }
                this.projectErrors[normalizedConfigName] = null;
            }

            if (isVersionTypeScript15 || isVersionTypeScript16) {
                let result = super.configFileToProjectOptions(configFilename);
                this.setProjectLevelError(result, configFilename);
                return result;
            }

            let configFileToProjectOptions: ProjectResult = this.configFileToProjectOptionsExt(configFilename);

            configFileToProjectOptions = copyOptionsWithResolvedFilesWithoutExtensions(configFileToProjectOptions, ts_impl);
            if (isLogEnabled) {
                serverLogger("Parse config result options: " + JSON.stringify(configFileToProjectOptions));
            }

            this.setProjectLevelError(configFileToProjectOptions, normalizedConfigName);

            return configFileToProjectOptions;
        }

        private setProjectLevelError(configFileToProjectOptions: ProjectResult, normalizedConfigName: string) {
            if (configFileToProjectOptions.errors) {
                let errors: ts.Diagnostic[] = configFileToProjectOptions.errors;
                if (errors.length > 0) {
                    let errorsForService: ts.server.protocol.Diagnostic[] = errors.map(el => {
                        return {
                            end: undefined,
                            start: undefined,
                            text: ts_impl.flattenDiagnosticMessageText(el.messageText, "\n")
                        }
                    });
                    this.projectErrors[normalizedConfigName] = errorsForService;

                    //back compatibility 1.8
                    configFileToProjectOptions.error = {errorMsg: errorsForService[0].text};
                }
            } else if (configFileToProjectOptions.error) {
                let error = configFileToProjectOptions.error;
                let errorMessage = error.errorMsg ? error.errorMsg : "Error parsing tsconfig";
                this.projectErrors[normalizedConfigName] = [{
                    text: errorMessage,
                    end: undefined,
                    start: undefined,
                }];
            }
        }


        /**
         * copy of super#configFileToProjectOptions()
         */
        configFileToProjectOptionsExt(configFilename: string): ProjectResult {

            if (isLogEnabled) {
                serverLogger("Parse config " + configFilename);
            }
            configFilename = ts_impl.normalizePath(configFilename);
            // file references will be relative to dirPath (or absolute)
            const dirPath = ts_impl.getDirectoryPath(configFilename);
            const contents = this.host.readFile(configFilename);
            const rawConfig: { config?: ts.server.ProjectOptions; error? } = ts_impl.parseConfigFileTextToJson(configFilename, contents);
            if (rawConfig.error) {
                if (isLogEnabled) {
                    serverLogger("Parse config error " + JSON.stringify(rawConfig.error));
                }

                return {succeeded: false, errors: [rawConfig.error]};
            }
            else {
                let parsedJsonConfig = rawConfig.config;
                const parsedCommandLine = ts_impl.parseJsonConfigFileContent(parsedJsonConfig, this.host, dirPath, /*existingOptions*/ {}, configFilename);

                if (parsedCommandLine.errors && (parsedCommandLine.errors.length > 0)) {
                    let result: ProjectResult = {succeeded: false, errors: parsedCommandLine.errors};
                    // if (parsedCommandLine.fileNames && parsedCommandLine.options) {
                    //     const projectOptions = this.createProjectOptions(parsedCommandLine, parsedJsonConfig);
                    //     result.projectOptions = projectOptions;
                    // }

                    return result;
                }
                else {
                    const projectOptions = this.createProjectOptionsForCommandLine(parsedCommandLine, parsedJsonConfig);

                    return {succeeded: true, projectOptions};
                }
            }
        }

        private createProjectOptionsForCommandLine(parsedCommandLine: ts.ParsedCommandLine, parsedJsonConfig: ts.server.ProjectOptions):ts.server.ProjectOptions {
            const projectOptions: ts.server.ProjectOptions = {
                files: parsedCommandLine.fileNames,
                compilerOptions: parsedCommandLine.options,
            };
            if ((<any>parsedCommandLine).wildcardDirectories) {
                projectOptions.wildcardDirectories = (<any>parsedCommandLine).wildcardDirectories;
            }

            if (parsedJsonConfig && parsedJsonConfig.compileOnSave === false) {
                projectOptions.compileOnSave = false;
            }

            return projectOptions;
        }
    }

    /**
     * copy formatDiag method (but we use 'TS' prefix)
     */
    function formatDiagnostic(fileName: string, project: ts.server.Project, diagnostic: ts.Diagnostic): ts.server.protocol.Diagnostic {
        return {
            start: project.compilerService.host.positionToLineOffset(fileName, diagnostic.start),
            end: project.compilerService.host.positionToLineOffset(fileName, diagnostic.start + diagnostic.length),
            text: "TS" + diagnostic.code + ":" + ts_impl.flattenDiagnosticMessageText(diagnostic.messageText, "\n")
        };
    }

    function fixNameWithProcessor(filename: string, onError?, contentRoot?: string, sourceRoot?: string): string {
        if (pathProcessor) {
            filename = pathProcessor.getExpandedPath(filename, contentRoot, sourceRoot, onError);
        }
        return filename;
    }


    let ideSession: ts.server.Session = new IDESession(ts_impl.sys, Buffer.byteLength, process.hrtime, logger);


    return ideSession;
}






