/*
 * Decompiled with CFR 0.152.
 */
package org.junit.platform.commons.util;

import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.junit.platform.commons.meta.API;
import org.junit.platform.commons.util.BlacklistedExceptions;
import org.junit.platform.commons.util.Preconditions;

@API(value=API.Usage.Internal)
class ClasspathScanner {
    private static final Logger LOG = Logger.getLogger(ClasspathScanner.class.getName());
    private static final String CLASS_FILE_SUFFIX = ".class";
    private static final String ROOT_PACKAGE_NAME = "";
    private static final String MALFORMED_CLASS_NAME_ERROR_MESSAGE = "Malformed class name";
    private final Supplier<ClassLoader> classLoaderSupplier;
    private final BiFunction<String, ClassLoader, Optional<Class<?>>> loadClass;

    ClasspathScanner(Supplier<ClassLoader> classLoaderSupplier, BiFunction<String, ClassLoader, Optional<Class<?>>> loadClass) {
        this.classLoaderSupplier = classLoaderSupplier;
        this.loadClass = loadClass;
    }

    boolean isPackage(String packageName) {
        Preconditions.notBlank(packageName, "package name must not be null or blank");
        try {
            return this.getClassLoader().getResources(ClasspathScanner.packagePath(packageName)).hasMoreElements();
        }
        catch (Exception ex) {
            return false;
        }
    }

    List<Class<?>> scanForClassesInPackage(String basePackageName, Predicate<Class<?>> classFilter) {
        Preconditions.notBlank(basePackageName, "basePackageName must not be null or blank");
        List<File> dirs = this.allSourceDirsForPackage(basePackageName);
        return this.allClassesInSourceDirs(dirs, basePackageName, classFilter);
    }

    List<Class<?>> scanForClassesInClasspathRoot(File root, Predicate<Class<?>> classFilter) {
        Preconditions.notNull(root, "root must not be null");
        Preconditions.condition(root.isDirectory(), () -> "root must be an existing directory: " + root.getAbsolutePath());
        return this.findClassesInSourceDirRecursively(ROOT_PACKAGE_NAME, root, classFilter);
    }

    private List<Class<?>> allClassesInSourceDirs(List<File> sourceDirs, String basePackageName, Predicate<Class<?>> classFilter) {
        ArrayList classes = new ArrayList();
        for (File sourceDir : sourceDirs) {
            classes.addAll(this.findClassesInSourceDirRecursively(basePackageName, sourceDir, classFilter));
        }
        return classes;
    }

    private List<File> allSourceDirsForPackage(String basePackageName) {
        try {
            Enumeration<URL> resources = this.getClassLoader().getResources(ClasspathScanner.packagePath(basePackageName));
            ArrayList<File> dirs = new ArrayList<File>();
            while (resources.hasMoreElements()) {
                URL resource = resources.nextElement();
                dirs.add(new File(resource.getFile()));
            }
            return dirs;
        }
        catch (Exception ex) {
            return Collections.emptyList();
        }
    }

    private List<Class<?>> findClassesInSourceDirRecursively(String packageName, File sourceDir, Predicate<Class<?>> classFilter) {
        Preconditions.notNull(classFilter, "classFilter must not be null");
        ArrayList classes = new ArrayList();
        this.collectClassesRecursively(packageName, sourceDir, classFilter, classes);
        return classes;
    }

    private void collectClassesRecursively(String packageName, File sourceDir, Predicate<Class<?>> classFilter, List<Class<?>> classes) {
        File[] files = sourceDir.listFiles();
        if (files == null) {
            return;
        }
        for (File file : files) {
            if (ClasspathScanner.isClassFile(file)) {
                this.processClassFileSafely(packageName, file, classFilter, classes);
                continue;
            }
            if (!file.isDirectory()) continue;
            this.collectClassesRecursively(this.appendSubpackageName(packageName, file.getName()), file, classFilter, classes);
        }
    }

    private void processClassFileSafely(String packageName, File classFile, Predicate<Class<?>> classFilter, List<Class<?>> classes) {
        Optional<Class<?>> clazz = Optional.empty();
        try {
            clazz = this.loadClassFromFile(packageName, classFile);
            clazz.filter(classFilter).ifPresent(classes::add);
        }
        catch (InternalError internalError) {
            this.handleInternalError(classFile, clazz, internalError);
        }
        catch (Throwable throwable) {
            this.handleThrowable(classFile, throwable);
        }
    }

    private Optional<Class<?>> loadClassFromFile(String packageName, File classFile) {
        String className = packageName + '.' + classFile.getName().substring(0, classFile.getName().length() - CLASS_FILE_SUFFIX.length());
        return this.loadClass.apply(className, this.getClassLoader());
    }

    private void handleInternalError(File classFile, Optional<Class<?>> clazz, InternalError ex) {
        if (MALFORMED_CLASS_NAME_ERROR_MESSAGE.equals(ex.getMessage())) {
            this.logMalformedClassName(classFile, clazz, ex);
        } else {
            this.logGenericFileProcessingException(classFile, ex);
        }
    }

    private void handleThrowable(File classFile, Throwable throwable) {
        BlacklistedExceptions.rethrowIfBlacklisted(throwable);
        this.logGenericFileProcessingException(classFile, throwable);
    }

    private void logMalformedClassName(File classFile, Optional<Class<?>> clazz, InternalError ex) {
        try {
            if (clazz.isPresent()) {
                ClasspathScanner.logWarning(ex, () -> String.format("The java.lang.Class loaded from file [%s] has a malformed class name [%s].", classFile.getAbsolutePath(), ((Class)clazz.get()).getName()));
            } else {
                ClasspathScanner.logWarning(ex, () -> String.format("The java.lang.Class loaded from file [%s] has a malformed class name.", classFile.getAbsolutePath()));
            }
        }
        catch (Throwable t) {
            ex.addSuppressed(t);
            this.logGenericFileProcessingException(classFile, ex);
        }
    }

    private void logGenericFileProcessingException(File classFile, Throwable throwable) {
        ClasspathScanner.logWarning(throwable, () -> String.format("Failed to load java.lang.Class for file [%s] during classpath scanning.", classFile.getAbsolutePath()));
    }

    private String appendSubpackageName(String packageName, String subpackageName) {
        return !packageName.isEmpty() ? packageName + "." + subpackageName : subpackageName;
    }

    private ClassLoader getClassLoader() {
        return this.classLoaderSupplier.get();
    }

    private static boolean isClassFile(File file) {
        return file.isFile() && file.getName().endsWith(CLASS_FILE_SUFFIX);
    }

    private static String packagePath(String packageName) {
        return packageName.replace('.', '/');
    }

    private static void logWarning(Throwable throwable, Supplier<String> msgSupplier) {
        LOG.log(Level.WARNING, throwable, msgSupplier);
    }
}

