/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.spring.model.jam;

import com.intellij.jam.annotations.JamPsiConnector;
import com.intellij.jam.annotations.JamPsiValidity;
import com.intellij.psi.*;
import com.intellij.psi.util.PsiTypesUtil;
import com.intellij.psi.util.PsiUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public abstract class JamPsiMethodSpringBean extends JamPsiMemberSpringBean<PsiMethod> {

  @NotNull
  @JamPsiConnector
  public abstract PsiMethod getPsiElement();

  @Override
  @JamPsiValidity
  public abstract boolean isValid();

  @Nullable
  public String getBeanName() {
    return getPsiElement().getName();
  }

  @Nullable
  public PsiClass getBeanClass() {
    final PsiMethod method = getPsiElement();
    return PsiTypesUtil.getPsiClass(method.getReturnType());
  }

  @Nullable
  public PsiType getBeanType() {
    return getPsiElement().getReturnType();
  }

  @Nullable
  @Override
  public PsiClass getBeanClass(boolean considerFactories) {
    final PsiType type = getBeanType(considerFactories);
    return type != null ? PsiTypesUtil.getPsiClass(type) : null;
  }

  /**
   * Returns actual {@code new MyImpl} type taken from method implementation if {@code considerFactories = true}.
   *
   * @since 16
   */
  @Nullable
  @Override
  public PsiType getBeanType(boolean considerFactories) {
    if (!considerFactories) {
      return getBeanType();
    }

    final PsiReturnStatement[] returnStatements = PsiUtil.findReturnStatements(getPsiElement());
    if (returnStatements.length == 1) {
      final PsiExpression returnValue = returnStatements[0].getReturnValue();

      if (returnValue != null) {
        return returnValue.getType();
      }
    }
    return getBeanType();
  }
}