/**
 * Entry point for the TypeScript plugin
 */

import {getService} from "./service-loader";
import {getSession} from "./ts-session";
import {createLoggerFromEnv} from "./logger-impl";
import {getPathProcessor, PathProcessor} from "./out-path-process";

class TypeScriptLanguagePlugin implements LanguagePlugin {

    private _session: ts.server.Session;
    readyMessage: {version: string, supportedErrorCodes?:(string|number)[]};

    constructor(state: TypeScriptPluginState) {

        let serviceInfo = getService(state.serverFolderPath);
        let serviceContext: {
            ts: typeof ts;
        } = serviceInfo.context;


        let serverFilePath = serviceInfo.serverFilePath;
        let ts_impl: typeof ts = serviceContext.ts;
        let loggerImpl = createLoggerFromEnv(ts_impl);
        overrideSysDefaults(ts_impl, serverFilePath);

        let commonDefaultCommandLine: ts.ParsedCommandLine = state.hasManualParams && state.commandLineArguments ?
            ts_impl.parseCommandLine(state.commandLineArguments) :
            null;

        let commonDefaultOptions: ts.CompilerOptions = null;
        if (commonDefaultCommandLine != null) {
            commonDefaultOptions = commonDefaultCommandLine.options;
        }

        if (commonDefaultOptions === null && state.hasManualParams) {
            commonDefaultOptions = undefined;
        }


        let pathProcessor: PathProcessor = null;
        if (state.hasManualParams && state.outPath) {
            pathProcessor = getPathProcessor(ts_impl, state);
        }

        let mainFile: string = null;
        if (state.hasManualParams && state.mainFilePath) {
            mainFile = state.mainFilePath;
        }

        this._session = getSession(ts_impl, loggerImpl, commonDefaultOptions, pathProcessor, mainFile);



        this.readyMessage = {version: ts_impl.version};

        if ((<any>ts_impl).getSupportedCodeFixes) {
            let codes:string[] = (<any>ts_impl).getSupportedCodeFixes();
            if (codes && codes.length >0)  {
                this.readyMessage.supportedErrorCodes = codes;
            }
        }
    }

    onMessage(p: string) {
        this._session.onMessage(p);
    }


}

class TypeScriptLanguagePluginFactory implements LanguagePluginFactory {
    create(state: PluginState): {languagePlugin: LanguagePlugin, readyMessage?:any } {
        let typeScriptLanguagePlugin = new TypeScriptLanguagePlugin(<TypeScriptPluginState>state);
        return {
            languagePlugin: typeScriptLanguagePlugin,
            readyMessage: typeScriptLanguagePlugin.readyMessage
        };
    }
}


function overrideSysDefaults(ts_impl: typeof ts, serverFolderPath: string) {
    const pending: string[] = [];
    let canWrite = true;

    function writeMessage(s: string) {
        if (!canWrite) {
            pending.push(s);
        }
        else {
            canWrite = false;
            process.stdout.write(new Buffer(s, "utf8"), setCanWriteFlagAndWriteMessageIfNecessary);
        }
    }

    function setCanWriteFlagAndWriteMessageIfNecessary() {
        canWrite = true;
        if (pending.length) {
            writeMessage(pending.shift());
        }
    }

    // Override sys.write because fs.writeSync is not reliable on Node 4
    ts_impl.sys.write = (s: string) => writeMessage(s);

    //ts 2.0 compatibility
    (<any>ts_impl.sys).setTimeout = setTimeout;
    (<any>ts_impl.sys).clearTimeout = clearTimeout;

    //ts2.0.5 & 2.1
    (<any>ts_impl.sys).setImmediate = setImmediate;
    (<any>ts_impl.sys).clearImmediate = clearImmediate;
    if (typeof global !== "undefined" && global.gc) {
        (<any>ts_impl.sys).gc = () => global.gc();
    }

    ts_impl.sys.getExecutingFilePath = () => {
        return serverFolderPath;
    }
}

let typescriptLanguagePluginFactory: TypeScriptLanguagePluginFactory = new TypeScriptLanguagePluginFactory();

export {typescriptLanguagePluginFactory}



