import {DiagnosticsContainer, IDETypeScriptSession, Response, doneRequest} from "./util";
import {HolderContainer} from "./compile-info-holder";
import {serverLogger, isLogEnabled} from "./logger-impl";
import {extendProjectServiceNew} from "./ts-project-service-new";


export function getSessionNew(TypeScriptSession: typeof IDETypeScriptSession,
                              TypeScriptProjectService: typeof ts.server.ProjectService,
                              TypeScriptCommandNames: typeof ts.server.CommandNames,
                              logger: ts.server.Logger,
                              host: ts.server.ServerHost,
                              ts_impl: typeof ts,
                              commonDefaultOptions: any,
                              mainFile: any,
                              projectEmittedWithAllFiles: HolderContainer,
                              getFileWrite: (project: string, outFiles: string[], contentRoot?: string, sourceRoot?: string) => any,
                              fixNameWithProcessor: (filename: string, onError?: any, contentRoot?: string, sourceRoot?: string) => string,
                              ensureDirectoriesExist: (p: string) => void): ts.server.Session {
    extendProjectServiceNew(TypeScriptProjectService, ts_impl, host, projectEmittedWithAllFiles);

    class IDESession extends TypeScriptSession {

        appendProjectErrors(result: ts.server.protocol.DiagnosticEventBody[], processedProjects: {[p: string]: ts.server.Project}, empty: boolean): ts.server.protocol.DiagnosticEventBody[] {
            if (!processedProjects) return result;

            for (let projectName in processedProjects) {
                if (processedProjects.hasOwnProperty(projectName)) {
                    let processedProject: ts.server.Project = processedProjects[projectName];
                    let errors: ts.Diagnostic[] = (<any>processedProject).getProjectErrors();
                    if (errors && errors.length > 0) {
                        let items = errors.map((el: ts.Diagnostic) => this.formatDiagnostic(projectName, processedProject, el));
                        result = result.concat({
                            file: projectName,
                            diagnostics: items
                        });
                    }

                }
            }

            return result;
        }


        beforeFirstMessage(): void {
            super.beforeFirstMessage();

            if (commonDefaultOptions !== null) {
                if (commonDefaultOptions.compileOnSave === undefined) {
                    commonDefaultOptions.compileOnSave = true;
                }
                this.getProjectService().setCompilerOptionsForInferredProjects(commonDefaultOptions);
            }
        }

        setNewLine(project: ts.server.Project, options: ts.CompilerOptions): void {
            //todo lsHost is a private field
            let host = (<any>project).lsHost;
            if (!host) {
                serverLogger("API was changed Project#lsHost is not found", true);
            }
            if (host && !host.getNewLine) {
                host.getNewLine = function () {
                    return ts_impl.getNewLineCharacter(options ? options : {});
                };
            }
        }

        getCompileOptionsEx(project: ts.server.Project): ts.CompilerOptions {
            return (<any>project).getCompilerOptions();
        }

        needRecompile(project: ts.server.Project): boolean {
            if (!project) return true;

            let compilerOptions = (<any>project).getCompilerOptions();
            if (compilerOptions && compilerOptions.___processed_marker) {
                return (<any>project).compileOnSaveEnabled;
            }

            return true;
        }

        afterCompileProcess(project: ts.server.Project, requestedFile: string, wasOpened: boolean): void {
            if (project) {
                let projectService = this.getProjectService();
                let externalProjects: ts.server.Project[] = projectService.externalProjects;
                for (let project of externalProjects) {
                    //close old projects
                    let projectName = this.getProjectName(project);
                    serverLogger("Close external project " + projectName, true);

                    projectService.closeExternalProject(projectName)
                }

                if (wasOpened) {
                    serverLogger("Close the opened file", true);
                    projectService.closeClientFile(requestedFile);
                }
            }
        }

        private isExternalProject(project: ts.server.Project) {
            let projectKind = (<any>project).projectKind;
            return projectKind && projectKind == (<any>ts_impl).server.ProjectKind.External;
        }

        getProjectForCompileRequest(req: ts.server.protocol.IDECompileFileRequestArgs, normalizedRequestedFile: string): {project: ts.server.Project; wasOpened: boolean} {
            if (req.file) {
                let projectService = this.getProjectService();
                let project: ts.server.Project;

                try {
                    let project = projectService.getDefaultProjectForFile(normalizedRequestedFile, true);
                } catch (e) {
                    //no project
                }
                if (project) {
                    return {project, wasOpened: false};
                }

                project = this.getFromExistingProject(normalizedRequestedFile);
                if (project) {
                    return {project, wasOpened: false};
                }

                let openClientFile: {
                    configFileName: string
                } = projectService.openClientFileWithNormalizedPath(normalizedRequestedFile);

                project = this.getFromExistingProject(normalizedRequestedFile);

                if (project && openClientFile && openClientFile.configFileName) {
                    projectService.openExternalProject({
                        projectFileName: openClientFile.configFileName,
                        rootFiles: [{fileName: openClientFile.configFileName}],
                        options: {}
                    })

                    projectService.closeClientFile(normalizedRequestedFile);

                    serverLogger("Project was created for compiling", true);

                    //keep project
                    return {project, wasOpened: false};
                } else {

                    serverLogger("File was opened for compiling", true);
                    return {project, wasOpened: true};
                }


            } else if (req.tsConfig) {
                let projectService = this.getProjectService();
                let configProject = projectService.findProject(normalizedRequestedFile);
                if (configProject) {
                    return {project: configProject, wasOpened: false};
                }

                serverLogger("Open external project", true);
                projectService.openExternalProject({
                    projectFileName: normalizedRequestedFile,
                    rootFiles: [{fileName: normalizedRequestedFile}],
                    options: {}
                })

                return {project: projectService.findProject(normalizedRequestedFile), wasOpened: false}
            }

            return {project: null, wasOpened: false};
        }

        positionToLineOffset(project: ts.server.Project, fileName: string, position: number): ts.server.ILineInfo {

            //todo review performance
            let scriptInfo = this.getProjectService().getScriptInfo(fileName);
            if (!scriptInfo) {
                serverLogger("ERROR! Cannot find script info for file " + fileName, true);
                return undefined;
            }

            return scriptInfo.positionToLineOffset(position);
        }

        containsFileEx(project: ts.server.Project, file: string, reqOpen: boolean): boolean {
            return (<any>project).containsFile(file, reqOpen);
        }


        getProjectName(project: ts.server.Project): string|any|any {
            return (<any>project).getProjectName();
        }

        getProjectConfigPathEx(project: ts.server.Project): string|any {
            if (this.isExternalProject(project)) {
                return this.getProjectName(project);
            }

            let configFileName = (<any>project).configFileName;
            return configFileName;
        }

        executeCommand(request: ts.server.protocol.Request): Response {
            let startTime = this.getTime();
            let command = request.command;
            try {
                if (TypeScriptCommandNames.Open == command || TypeScriptCommandNames.Close == command) {
                    super.executeCommand(request);

                    //open | close command doesn't send answer so we have to override
                    return doneRequest;
                } else if (TypeScriptCommandNames.ReloadProjects == command) {
                    projectEmittedWithAllFiles.reset();
                    this.updateProjectStructureEx();
                    return doneRequest;
                } else if (TypeScriptCommandNames.IDEChangeFiles == command) {
                    const updateFilesArgs: ts.server.protocol.IDEUpdateFilesContentArgs = <ts.server.protocol.IDEUpdateFilesContentArgs>request.arguments;
                    return this.updateFilesEx(updateFilesArgs);
                } else if (TypeScriptCommandNames.IDECompile == command) {
                    const fileArgs = <ts.server.protocol.IDECompileFileRequestArgs>request.arguments;
                    return this.compileFileEx(fileArgs);
                } else if (TypeScriptCommandNames.IDECompletions == command) {
                    return this.getCompletionEx(request);
                } else if (TypeScriptCommandNames.IDEGetErrors == command) {
                    let args = request.arguments;
                    return {response: {infos: this.getDiagnosticsEx(args.files)}, responseRequired: true}
                } else if (TypeScriptCommandNames.IDEGetMainFileErrors == command) {
                    let args: ts.server.protocol.FileRequestArgs = request.arguments;
                    return {response: {infos: this.getMainFileDiagnosticsForFileEx(args.file)}, responseRequired: true}
                } else if (TypeScriptCommandNames.IDEGetProjectErrors == command) {
                    let args: ts.server.protocol.FileRequestArgs = request.arguments;
                    let projectDiagnosticsForFileEx = this.getProjectDiagnosticsForFileEx(args.file);
                    return {response: {infos: projectDiagnosticsForFileEx}, responseRequired: true}
                }

                return super.executeCommand(request);
            } finally {
                let processingTime = Date.now() - startTime;
                serverLogger("Message " + request.seq + " '" + command + "' server time, mills: " + processingTime, true);
            }
        }

        getProjectForFileEx(fileName: string, projectFile?: string): ts.server.Project {
            if (!projectFile) {
                fileName = ts_impl.normalizePath(fileName);
                return this.getProjectService().getDefaultProjectForFile(fileName, true);
            }

            return this.getProjectService().findProject(projectFile);
        }

        closeClientFileEx(normalizedFileName: string): void {
            let scriptInfoForNormalizedPath = this.getProjectService().getScriptInfoForNormalizedPath(normalizedFileName);
            if (!scriptInfoForNormalizedPath || !scriptInfoForNormalizedPath.isOpen) {
                return;
            }
            this.projectService.closeClientFile(normalizedFileName);
        }

        configFileDiagnosticEvent(triggerFile: string, configFile: string, errors: ts.Diagnostic[]) {

        }

        refreshStructureEx(): void {
            this.getProjectService().refreshInferredProjects();
        }

        changeFileEx(fileName: string, content: string, tsconfig?: string): void {
            fileName = ts_impl.normalizePath(fileName);

            let info = this.getProjectService().getScriptInfoForNormalizedPath(fileName);
            if (info && info.isOpen) {
                this.getProjectService().getOrCreateScriptInfo(fileName, true, content);
            } else {
                this.getProjectService().openClientFileWithNormalizedPath(fileName, content);
            }
        }

        getLanguageService(project: ts.server.Project): ts.LanguageService {
            return (<any>project).getLanguageService();
        }

        public event(info: any, eventName: string) {
            if (isLogEnabled) {
                serverLogger("Event " + eventName);
            }
        }

        lineOffsetToPosition(project: ts.server.Project, fileName: string, line: number, offset: number): undefined {
            //todo review performance
            let scriptInfo = this.getProjectService().getScriptInfo(fileName);
            if (!scriptInfo) {
                serverLogger("ERROR! Cannot find script info for file " + fileName, true);
                return undefined;
            }

            return scriptInfo.lineOffsetToPosition(line, offset);
        }

        /**
         * todo change d.ts files & replace any by ts.server.ProjectService
         */
        private getProjectService(): any {
            return this.projectService;
        }

        private getFromExistingProject(normalizedRequestedFile: string) {
            let projectService = this.getProjectService();

            {
                let inferredProjects: ts.server.Project[] = projectService.inferredProjects;
                for (let project of inferredProjects) {
                    if (this.containsFileEx(project, normalizedRequestedFile, false)) {
                        return project;
                    }
                }
            }

            {
                let configuredProjects: ts.server.Project[] = projectService.configuredProjects;

                for (let project of configuredProjects) {
                    if (this.containsFileEx(project, normalizedRequestedFile, false)) {
                        return project;
                    }
                }
            }

            return null;
        }
    }


    let cancellationToken: any;
    try {
        const factory = require("./cancellationToken");
        cancellationToken = factory(host.args);
    }
    catch (e) {
        cancellationToken = {
            isCancellationRequested: () => false
        };
    }

    const nullTypingsInstaller = {
        enqueueInstallTypingsRequest: () => {
        },
        attach: (projectService: any) => {
        },
        onProjectClosed: (p: any) => {
        },
        globalTypingsCacheLocation: <any>undefined
    };

    //todo remove after replacing typing
    let IDESessionImpl: any = IDESession;

    let session = new IDESessionImpl(host, cancellationToken, false, nullTypingsInstaller, Buffer.byteLength, process.hrtime, logger, true);

    return session;
}