/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.dom;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.util.ArrayUtil;
import com.intellij.util.xml.DomFileDescription;
import com.intellij.util.xml.reflect.DomExtensionsRegistrar;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Allows registration of custom namespace(s) in Spring DOM.
 */
public abstract class SpringCustomNamespaces {

  public static final ExtensionPointName<SpringCustomNamespaces> EP_NAME =
    ExtensionPointName.create("com.intellij.spring.customNamespaces");

  public abstract NamespacePolicies getNamespacePolicies();

  /**
   * Convenience method to register namespace DOM elements.
   * <p/>
   * Alternatively, one can use custom {@link com.intellij.util.xml.reflect.DomExtender}.
   *
   * @param registrar Registrar instance.
   * @see CustomNamespaceRegistrar
   * @since 14
   */
  public void registerExtensions(DomExtensionsRegistrar registrar) {
  }

  /**
   * Returns DOM model version for all namespaces registered in this EP.
   * <p/>
   * Must be incremented on any Spring model related change
   * (e.g. changing {@link com.intellij.spring.model.xml.BeanType} or
   * linked {@link com.intellij.spring.model.xml.BeanTypeProvider},
   * custom implementation of {@link com.intellij.spring.model.xml.DomSpringBeanImpl#getBeanName()})
   * to update all related indexes.
   *
   * @return Model version.
   */
  public int getModelVersion() {
    return 0;
  }

  /**
   * Returns DOM stub version for all namespaces registered in this EP.
   * <p/>
   * Must be incremented on any change of using {@link com.intellij.util.xml.Stubbed} in related DOM.
   *
   * @return 0 if no stubs are used (default).
   * @see com.intellij.util.xml.Stubbed
   */
  public int getStubVersion() {
    return 0;
  }


  /**
   * @see DomFileDescription#registerNamespacePolicy(String, String...)
   */
  public static class NamespacePolicies {
    private final Map<String, List<String>> policies = new HashMap<>();

    public static NamespacePolicies simple(String key, String namespace) {
      NamespacePolicies policies = new NamespacePolicies();
      return policies.add(key, namespace);
    }

    public NamespacePolicies add(String key, String... namespaces) {
      policies.put(key, Arrays.asList(namespaces));
      return this;
    }

    void process(DomFileDescription domFileDescription) {
      for (Map.Entry<String, List<String>> entry : policies.entrySet()) {
        domFileDescription.registerNamespacePolicy(entry.getKey(), ArrayUtil.toStringArray(entry.getValue()));
      }
    }
  }
}
