/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.model.jam.testContexts;

import com.intellij.jam.JamSimpleReferenceConverter;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.openapi.module.Module;
import com.intellij.psi.*;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.FileReference;
import com.intellij.psi.xml.XmlFile;
import com.intellij.spring.model.converters.SpringImportResourceConverter;
import com.intellij.spring.model.utils.resources.SpringResourcesBuilder;
import com.intellij.spring.model.utils.resources.SpringResourcesUtil;
import com.intellij.util.Function;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.NullableFunction;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.List;

/**
 * @since 2017.1
 */
public abstract class ResourcePathReferenceConverter <T> extends JamSimpleReferenceConverter<List<T>> {

  @Override
  public List<T> fromString(@Nullable String s, JamStringAttributeElement<List<T>> context) {
    return s == null ? null : getFiles(getReferences(s, context));
  }


  public List<T> getFiles(PsiReference[] references) {
    for (PsiReference reference : references) {
      final FileReference fileReference = FileReference.findFileReference(reference);
      if (fileReference == null) continue;

      FileReference lastFileReference = fileReference.getLastFileReference();
      if (lastFileReference != null) {
        ResolveResult[] resolve = lastFileReference.multiResolve(false);
        return ContainerUtil.mapNotNull(resolve, getMapper());
      }
    }
    return Collections.emptyList();
  }

  protected abstract  Function<ResolveResult, T > getMapper();

  @NotNull
  @Override
  public PsiReference[] createReferences(JamStringAttributeElement<List<T>> context) {
    return getReferences(context);
  }

  protected   PsiReference[] getReferences(@NotNull String s, JamStringAttributeElement<List<T>> context) {
    PsiLiteral psiLiteral = context.getPsiLiteral();
    if (psiLiteral == null) {
      return getReferencesOnFakeElement(context.getPsiElement(), s);
    }

    return getReferences(psiLiteral, s);
  }

  private static PsiReference[] getReferences(PsiLiteral psiLiteral, String s, Module... modules) {
    final SpringResourcesBuilder resourcesBuilder =
      SpringResourcesBuilder.create(psiLiteral, s).fromRoot(s.startsWith("/")).soft(false).modules(modules);
    return SpringResourcesUtil.getInstance().getReferences(resourcesBuilder);
  }

  @NotNull
  protected static PsiReference[] getReferencesOnFakeElement(@Nullable PsiElement element, @NotNull String s, Module... modules) {
    if (element == null) {
      return PsiReference.EMPTY_ARRAY;
    }
    try {
      final PsiExpression psiExpression =
        JavaPsiFacade.getElementFactory(element.getProject()).createExpressionFromText("\"" + s + "\"", element);
      if (psiExpression instanceof PsiLiteral) {
        return getReferences((PsiLiteral)psiExpression, s, modules);
      }
    }
    catch (IncorrectOperationException e) {
      // ignore
    }
    return PsiReference.EMPTY_ARRAY;
  }

  private  PsiReference[] getReferences(JamStringAttributeElement<List<T>> jamAttribute) {
    PsiLiteral psiLiteral = jamAttribute.getPsiLiteral();
    if (psiLiteral == null) return PsiReference.EMPTY_ARRAY;

    String s = ElementManipulators.getValueText(psiLiteral);
    return getReferences(psiLiteral, s);
  }
}
