import {onMessage15, reload15, close15, openClientFileConfig15, openProjectByConfig, setGetFileNames} from "./ts15impl";
import {Response, doneRequest, isTypeScript17, isTypeScript16, isTypeScript15, DiagnosticsContainer, IDETypeScriptSession} from "./util";
import {serverLogger, isLogEnabled} from "./logger-impl";
import {CompileInfoHolder, HolderContainer} from "./compile-info-holder";
import {extendProjectService} from "./ts-project-service-old";


export function getSessionOld(TypeScriptSession: typeof IDETypeScriptSession,
                              TypeScriptProjectService: typeof ts.server.ProjectService,
                              TypeScriptCommandNames: typeof ts.server.CommandNames,
                              logger: ts.server.Logger,
                              host: ts.server.ServerHost,
                              ts_impl: typeof ts,
                              commonDefaultOptions: any,
                              mainFile: any,
                              projectEmittedWithAllFiles: HolderContainer): ts.server.Session {
    let projectErrors: DiagnosticsContainer = new DiagnosticsContainer();

    let isVersionTypeScript15 = isTypeScript15(ts_impl);
    let isVersionTypeScript16 = isTypeScript16(ts_impl);
    let isVersionTypeScript17 = isTypeScript17(ts_impl);

    if (isVersionTypeScript15) {
        setGetFileNames(ts_impl.server.Project);
    }

    extendProjectService(
        TypeScriptProjectService,
        ts_impl,
        host,
        projectEmittedWithAllFiles,
        projectErrors,
        isVersionTypeScript15,
        isVersionTypeScript16,
        isVersionTypeScript17,
        commonDefaultOptions);


    class IDESession extends TypeScriptSession {

        afterCompileProcess(project: ts.server.Project, requestedFile: string, wasOpened: boolean): void {
        }

        getProjectName(project: ts.server.Project): string|any|any {
            return project.projectFilename;
        }

        getProjectConfigPathEx(project: ts.server.Project): string|any {
            return project.projectFilename;
        }


        positionToLineOffset(project: ts.server.Project, fileName: string, position: number): ts.server.ILineInfo {
            return project.compilerService.host.positionToLineOffset(fileName, position);
        }

        containsFileEx(project: ts.server.Project, file: string, reqOpen: boolean): boolean {
            return project.getSourceFileFromName(file, reqOpen) != null;
        }

        lineOffsetToPosition(project: ts.server.Project, fileName: string, line: number, offset: number): number {
            return project.compilerService.host.lineOffsetToPosition(fileName, line, offset);
        }

        getLanguageService(project: ts.server.Project): ts.LanguageService {
            return project.compilerService.languageService;
        }

        beforeFirstMessage(): void {
            super.beforeFirstMessage();
        }

        tsVersion(): string {
            return "2.0.0";
        }

        constructor(byteLength: (buf: string, encoding?: string) => number,
                    hrtime: (start?: number[]) => number[],
                    logger: ts.server.Logger) {
            if (isVersionTypeScript15) {
                super(host, <any>logger, undefined, undefined);
            } else {
                super(host, byteLength, hrtime, logger);
            }
        }


        onMessage(message: string): void {
            if (isVersionTypeScript15) {
                onMessage15(this, message);
                return;
            }
            super.onMessage(message);
        }

        executeCommand(request: ts.server.protocol.Request): Response {
            let startTime = Date.now();
            let command = request.command;
            try {
                if (TypeScriptCommandNames.Open == command) {
                    //use own implementation
                    const openArgs = <ts.server.protocol.OpenRequestArgs>request.arguments;
                    this.openClientFileEx(openArgs);
                    return doneRequest;

                } else if (TypeScriptCommandNames.ReloadProjects == command) {
                    projectEmittedWithAllFiles.reset();

                    if (isVersionTypeScript15) return reload15(this, ts_impl);
                    const requestWithConfigArgs = <ts.server.protocol.FileRequestWithConfigArgs>request.arguments;
                    if (requestWithConfigArgs.projectFileName) {
                        const configFileName = ts_impl.normalizePath(requestWithConfigArgs.projectFileName);
                        let project = this.projectService.findConfiguredProjectByConfigFile(configFileName);
                        if (project != null) {
                            this.projectService.updateConfiguredProject(project);
                        }
                    }

                    this.refreshStructureEx();
                    return doneRequest;
                } else if (TypeScriptCommandNames.IDEChangeFiles == command) {
                    const updateFilesArgs: ts.server.protocol.IDEUpdateFilesContentArgs = <ts.server.protocol.IDEUpdateFilesContentArgs>request.arguments;
                    return this.updateFilesEx(updateFilesArgs);
                } else if (TypeScriptCommandNames.IDECompile == command) {
                    const fileArgs = <ts.server.protocol.IDECompileFileRequestArgs>request.arguments;
                    return this.compileFileEx(fileArgs);
                } else if (TypeScriptCommandNames.Close == command) {

                    if (isVersionTypeScript15) {
                        close15(this, request);
                        return doneRequest;
                    }

                    super.executeCommand(request);
                    return doneRequest;
                } else if (TypeScriptCommandNames.IDEGetErrors == command) {
                    let args = request.arguments;
                    return {response: {infos: this.getDiagnosticsEx(args.files)}, responseRequired: true}
                } else if (TypeScriptCommandNames.IDEGetMainFileErrors == command) {
                    let args: ts.server.protocol.FileRequestArgs = request.arguments;
                    return {response: {infos: this.getMainFileDiagnosticsForFileEx(args.file)}, responseRequired: true}
                } else if (TypeScriptCommandNames.IDEGetProjectErrors == command) {
                    let args: ts.server.protocol.FileRequestArgs = request.arguments;
                    return {response: {infos: this.getProjectDiagnosticsForFileEx(args.file)}, responseRequired: true}
                } else if (TypeScriptCommandNames.IDECompletions == command) {
                    if (isVersionTypeScript15) return {response: [], responseRequired: true};

                    return this.getCompletionEx(request);
                }

                return super.executeCommand(request);
            } finally {
                let processingTime = Date.now() - startTime;

                serverLogger("Message " + request.seq + " '" + command + "' server time, mills: " + processingTime, true);
            }
        }


        closeClientFileEx(normalizedFileName: string) {
            this.projectService.closeClientFile(normalizedFileName);
        }

        refreshStructureEx(): void {
            super.refreshStructureEx();
            this.projectService.updateProjectStructure();
        }

        openClientFileEx(openArgs: ts.server.protocol.OpenRequestArgs): {response?: any; responseRequired?: boolean} {
            const fileName = openArgs.file;
            const fileContent = openArgs.fileContent;
            const configFile = openArgs.projectFileName;
            const file = ts_impl.normalizePath(fileName);

            return this.openFileEx(file, fileContent, configFile);
        }

        openFileEx(fileName: string, fileContent: string, configFileName: string): {response?: any; responseRequired?: boolean} {
            let projectService = this.projectService;
            if (configFileName) {
                projectService.openOrUpdateConfiguredProjectForFile(ts_impl.normalizePath(configFileName));
            } else {
                if (isVersionTypeScript15)  return openClientFileConfig15(projectService, fileName, fileContent, ts_impl);

                projectService.openOrUpdateConfiguredProjectForFile(fileName);
            }

            const info: ts.server.ScriptInfo = projectService.openFile(fileName, /*openedByClient*/ true, fileContent);

            projectService.addOpenFile(info);
            return info;
        }

        changeFileEx(fileName: string, content: string, tsconfig?: string) {
            const file = ts_impl.normalizePath(fileName);
            const project = this.getForceProject(file);
            if (project) {
                const compilerService = project.compilerService;
                let scriptInfo = compilerService.host.getScriptInfo(file);
                if (scriptInfo != null) {
                    scriptInfo.svc.reload(content);
                    if (isLogEnabled) {
                        serverLogger("Update file reload content from text " + file);
                    }
                } else {
                    if (isLogEnabled) {
                        serverLogger("Update file scriptInfo is null " + file);
                    }

                    this.openClientFileEx({
                        file: fileName,
                        fileContent: content,
                        projectFileName: tsconfig
                    });
                }
            } else {
                if (isLogEnabled) {
                    serverLogger("Update file cannot find project for " + file);
                }
                this.openClientFileEx({
                    file: fileName,
                    fileContent: content,
                    projectFileName: tsconfig
                });
            }
        }


        getProjectForCompileRequest(req: ts.server.protocol.IDECompileFileRequestArgs, normalizedRequestedFile: string): {project: ts.server.Project; wasOpened: boolean} {
            let project: ts.server.Project = null;
            if (req.file) {
                project = this.projectService.getProjectForFile(normalizedRequestedFile);
                if (project) {
                    return {project, wasOpened: false};
                }

                //by some reason scriptInfo can exist but defaultProject == null
                //lets try to detect project by opened files (if they are exist)
                let scriptInfo = this.projectService.filenameToScriptInfo[normalizedRequestedFile];
                if (scriptInfo) {
                    //lets try to file parent project
                    project = getProjectForPath(scriptInfo, normalizedRequestedFile, this.projectService);
                    if (project) {
                        return {project, wasOpened: false};
                    }
                }

                this.projectService.openOrUpdateConfiguredProjectForFile(normalizedRequestedFile);
                return {project: this.projectService.getProjectForFile(normalizedRequestedFile), wasOpened: false};
            } else {
                if (isVersionTypeScript15)  return {
                    project: openProjectByConfig(this.projectService, normalizedRequestedFile, ts_impl),
                    wasOpened: false
                };

                project = this.projectService.findConfiguredProjectByConfigFile(normalizedRequestedFile);
                if (!project) {
                    const configResult = this.projectService.openConfigFile(normalizedRequestedFile);
                    if (configResult && configResult.project) {
                        return {project: configResult.project, wasOpened: false}
                    }
                }

            }
            return {project, wasOpened: false};
        }

        getProjectForFileEx(fileName: string, projectFile?: string): ts.server.Project| null {
            fileName = ts_impl.normalizePath(fileName);

            if (!projectFile) {
                return this.projectService.getProjectForFile(fileName);
            }


            projectFile = ts_impl.normalizePath(projectFile);


            return this.projectService.findConfiguredProjectByConfigFile(projectFile);
        }

        getForceProject(normalizedFileName: string): ts.server.Project| null {
            let projectForFileEx = this.getProjectForFileEx(normalizedFileName);
            if (projectForFileEx) {
                return projectForFileEx;
            }

            let configFile = this.projectService.findConfigFile(normalizedFileName);
            if (!configFile) {
                this.logMessage("Cannot find config file for this file " + normalizedFileName);
                return null;
            }
            return this.projectService.findConfiguredProjectByConfigFile(configFile);
        }

        needRecompile(project: ts.server.Project) {
            return !project || !project.projectOptions || project.projectOptions.compileOnSave || project.projectOptions.compileOnSave === undefined;
        }

        setNewLine(project: ts.server.Project, options: ts.CompilerOptions) {
            (<any>project.compilerService.host).getNewLine = function () {
                return ts_impl.getNewLineCharacter(options ? options : {});
            };
        }

        getCompileOptionsEx(project: ts.server.Project) {
            return project.projectOptions ? project.projectOptions.compilerOptions : null;
        }

        appendGlobalErrors(result: ts.server.protocol.DiagnosticEventBody[],
                           processedProjects: {[p: string]: ts.server.Project},
                           empty: boolean): ts.server.protocol.DiagnosticEventBody[] {
            result = super.appendGlobalErrors(result, processedProjects, empty);

            let values = projectErrors.value;
            for (let projectName in values) {
                if (values.hasOwnProperty(projectName)) {
                    let processedProjectErrors = values[projectName];
                    if (processedProjectErrors && processedProjectErrors.length > 0) {
                        if (empty || processedProjects[projectName]) {
                            let diagnosticBody: ts.server.protocol.DiagnosticEventBody = {
                                file: projectName,
                                diagnostics: processedProjectErrors
                            }
                            result = [diagnosticBody].concat(result);
                        }
                    }
                }
            }

            return result;
        }

    }

    function getProjectForPath(scriptInfo: ts.server.ScriptInfo, path: string, projectService: ts.server.ProjectService) {
        const searchPath = ts_impl.getDirectoryPath(path);
        if (searchPath) {
            const configFileName = projectService.findConfigFile(searchPath);
            if (configFileName) {
                let project = projectService.findConfiguredProjectByConfigFile(configFileName);
                if (project && project.getSourceFile(scriptInfo)) {
                    return project;
                }
            }
        }

        return null;
    }


    let ideSession: ts.server.Session = new IDESession(Buffer.byteLength, process.hrtime, logger);

    return ideSession;
}

