import {HolderContainer, CompileInfoHolder} from "./compile-info-holder";
import {PathProcessor} from "./out-path-process";
import {
    initCommandNames,
    IDETypeScriptSession,
    doneRequest,
    Response,
    isFunctionKind,
    DETAILED_COMPLETION_COUNT,
    DETAILED_MAX_TIME
} from "./util";
import {isLogEnabled, serverLogger} from "./logger-impl";
import IDECompositeRequest = ts.server.protocol.IDECompositeRequest;

export type ProjectResult = {succeeded: boolean; projectOptions?: ts.server.ProjectOptions; error?: ts.server.ProjectOpenResult, errors?: any};
export function createSessionClass(ts_impl: any /*must be typeof ts */,
                                   logger: ts.server.Logger,
                                   commonDefaultOptions?: ts.CompilerOptions,
                                   pathProcessor?: PathProcessor,
                                   projectEmittedWithAllFiles?: HolderContainer,
                                   mainFile?: string): typeof IDETypeScriptSession {

    let TypeScriptSession: typeof ts.server.Session = ts_impl.server.Session;
    let TypeScriptCommandNames: typeof ts.server.CommandNames = ts_impl.server.CommandNames;
    initCommandNames(TypeScriptCommandNames);
    let host: ts.server.ServerHost = ts_impl.sys;

    let wasFirstMessage = false;

    let version = ts_impl.version;

    abstract class IDESession extends TypeScriptSession implements IDETypeScriptSession {
        private _mySeq: number;

        /**
         * 2.0.0 (and other old versions)
         * 2.0.5 (and higher)
         */
        abstract tsVersion(): string;

        abstract needRecompile(project: ts.server.Project): boolean

        abstract getProjectForCompileRequest(req: ts.server.protocol.IDECompileFileRequestArgs, normalizedRequestedFile: string): {project: ts.server.Project; wasOpened: boolean};

        abstract getProjectForFileEx(fileName: string, projectFile?: string): ts.server.Project;

        abstract getLanguageService(project: ts.server.Project, sync?: boolean): ts.LanguageService;

        abstract lineOffsetToPosition(project: ts.server.Project, fileName: string, line: number, offset: number): number;

        abstract positionToLineOffset(project: ts.server.Project, fileName: string, position: number): ts.server.ILineInfo;

        abstract containsFileEx(project: ts.server.Project, file: string, reqOpen: boolean): boolean;

        abstract getProjectName(project: ts.server.Project): string| null;

        abstract getProjectConfigPathEx(project: ts.server.Project): string| null;

        abstract changeFileEx(fileName: string, content: string, tsconfig?: string): void

        abstract closeClientFileEx(normalizedFileName: string): void

        abstract setNewLine(project: ts.server.Project, options: ts.CompilerOptions): void;

        abstract getCompileOptionsEx(project: ts.server.Project): ts.CompilerOptions;

        abstract afterCompileProcess(project: ts.server.Project, requestedFile: string, wasOpened: boolean): void;

        beforeFirstMessage(): void {
        }

        refreshStructureEx(): void {
        }

        onMessage(message: string): void {
            if (!wasFirstMessage) {
                serverLogger("TypeScript service version: " + ts_impl.version, true);
                this.beforeFirstMessage();
                wasFirstMessage = true;
            }

            super.onMessage(message);
        }

        send(msg: ts.server.protocol.Message) {
            const json = JSON.stringify(msg);
            host.write(json + "\n");
        }

        logError(err: Error, cmd: string) {
            const typedErr = <any>err;
            serverLogger("Error processing message: " + err.message + " " + typedErr.stack, true);

            super.logError(err, cmd);
        }

        logMessage(text: string, force: boolean = false): void {
            serverLogger(text, force);
        }

        getChangeSeq() {
            let anyThis: any = this;
            let superClassSeq = anyThis.changeSeq;
            if (typeof superClassSeq !== "undefined") {
                return superClassSeq;
            }
            serverLogger("WARN: Used own sequence implementation (can be slow)", true);
            return this._mySeq;
        }

        updateProjectStructureEx(): void {
            let mySeq = this.getChangeSeq();
            let matchSeq = (n: any) => n === mySeq;
            setTimeout(() => {
                if (matchSeq(this.getChangeSeq())) {
                    let startTime = Date.now();
                    this.refreshStructureEx();
                    serverLogger("Update project structure scheduler time, mills: " + (Date.now() - startTime), true);
                }
            }, 1500);
        }

        getTime() {
            return Date.now();
        }

        compileFileEx(req: ts.server.protocol.IDECompileFileRequestArgs): Response {
            let startCompile = Date.now();

            let compileExactFile: boolean = req.file != null;
            if (!compileExactFile && !req.projectFileName) {
                return doneRequest;
            }

            let requestedFile: string = ts_impl.normalizePath(req.file ? req.file : req.projectFileName);
            if (!requestedFile) {
                return doneRequest;
            }
            let {project, wasOpened} = this.getProjectForCompileRequest(req, requestedFile);

            if (isLogEnabled) {
                serverLogger("Compile get project end time: " + (Date.now() - startCompile));
            }
            const outFiles: string[] = [];
            let includeErrors = req.includeErrors;
            let diagnostics: ts.server.protocol.DiagnosticEventBody[] = includeErrors ? [] : undefined;
            let needCompile = ((req.force || this.needRecompile(project)));
            if (project && needCompile) {
                let projectFilename = this.getProjectName(project);
                let languageService = this.getLanguageService(project);
                let program = languageService.getProgram();
                if (isLogEnabled) {
                    serverLogger("Compile get source files end time: " + program.getSourceFiles().length + "(count): time, mills: " + (Date.now() - startCompile));
                    serverLogger("Compile project Filename: " + (projectFilename ? projectFilename : "no filename"));
                }
                let options = this.getCompileOptionsEx(project);
                serverLogger("Options: " + JSON.stringify(options));
                this.setNewLine(project, options);

                let compileInfoHolder: CompileInfoHolder = null;
                if (projectFilename) {
                    compileInfoHolder = projectEmittedWithAllFiles.value[projectFilename];
                    compileExactFile = false;
                    if (!compileInfoHolder) {
                        compileInfoHolder = new CompileInfoHolder(ts_impl);
                        projectEmittedWithAllFiles.value[projectFilename] = compileInfoHolder;
                    }
                } else {
                    compileExactFile = false;
                }

                let fileWriteCallback = getFileWrite(this.getProjectConfigPathEx(project), outFiles, req.contentRootForMacro, req.sourceRootForMacro);
                if (!compileExactFile) {
                    serverLogger("Compile all files using cache checking", true);

                    let toUpdateFiles: ts.SourceFile[] = [];
                    let rawSourceFiles = program.getSourceFiles();
                    rawSourceFiles.forEach((val) => {
                        if (!compileInfoHolder || compileInfoHolder.checkUpdateAndAddToCache(val)) {
                            toUpdateFiles.push(val);
                        }
                    });


                    let compilerOptions = program.getCompilerOptions();
                    let useOutFile = compilerOptions && (compilerOptions.outFile || compilerOptions.out);
                    if (toUpdateFiles.length > 0) {
                        if (toUpdateFiles.length == rawSourceFiles.length || useOutFile) {
                            let emitResult: ts.EmitResult = program.emit(undefined, fileWriteCallback);
                            diagnostics = this.appendEmitDiagnostics(project, emitResult, diagnostics);
                        } else {
                            toUpdateFiles.forEach((el) => {
                                let emitResult: ts.EmitResult = program.emit(el, fileWriteCallback);
                                diagnostics = this.appendEmitDiagnostics(project, emitResult, diagnostics);
                            })
                        }
                    }


                    serverLogger("Compile end emit files: " + (Date.now() - startCompile));
                } else {
                    let sourceFile: ts.SourceFile = program.getSourceFile(requestedFile);
                    if (sourceFile) {
                        if (!compileInfoHolder || compileInfoHolder.checkUpdateAndAddToCache(sourceFile)) {
                            let emitResult: ts.EmitResult = project.program.emit(sourceFile, fileWriteCallback);
                            diagnostics = this.appendEmitDiagnostics(project, emitResult, diagnostics);

                        }
                    } else {
                        serverLogger("Compile can't find source file: shouldn't be happened")
                    }
                }
            } else {
                if (project) {
                    serverLogger("Compile skip: compileOnSave = false", true);
                } else {
                    serverLogger("Compile can't find project: shouldn't be happened", true);
                }
            }

            if (includeErrors) {
                diagnostics = diagnostics.concat(compileExactFile ?
                    this.getDiagnosticsEx([requestedFile], project) :
                    this.getProjectDiagnosticsEx(project)
                );
            }

            this.afterCompileProcess(project, requestedFile, wasOpened);

            if (isLogEnabled) {
                serverLogger("Compile end get diagnostics time, mills: " + (this.getTime() - startCompile));
            }

            return {response: {generatedFiles: outFiles, infos: diagnostics}, responseRequired: true};
        }


        appendGlobalErrors(result: ts.server.protocol.DiagnosticEventBody[],
                           processedProjects: {[p: string]: ts.server.Project},
                           empty: boolean): ts.server.protocol.DiagnosticEventBody[] {
            return result;
        }

        updateFilesEx(args: ts.server.protocol.IDEUpdateFilesContentArgs): Response {
            let updated = false;
            let files = args.files;
            for (let fileName in files) {
                if (files.hasOwnProperty(fileName)) {
                    let content = files[fileName];
                    if (content !== undefined) {
                        this.changeFileEx(fileName, content);
                        updated = true;
                    }
                }
            }

            if (args.filesToReloadContentFromDisk) {
                for (let fileName of args.filesToReloadContentFromDisk) {
                    if (!fileName) {
                        continue;
                    }

                    let file = ts_impl.normalizePath(fileName);
                    this.closeClientFileEx(file);

                    if (isLogEnabled) {
                        serverLogger("Update file from disk (by 'filesToReloadContentFromDisk') " + file);
                    }
                    updated = true;
                }
            }

            if (updated) {
                this.updateProjectStructureEx();
            }

            return doneRequest;
        }

        getCompletionEx(request: ts.server.protocol.Request) {
            let startDate = -1;
            if (isLogEnabled) {
                startDate = Date.now();
            }

            const args = <ts.server.protocol.CompletionsRequestArgs>request.arguments;

            let result: Response = super.executeCommand({
                command: TypeScriptCommandNames.Completions,
                arguments: args,
                seq: request.seq,
                type: request.type
            });


            if (isLogEnabled) {
                serverLogger("Completion service implementation time, mills: " + (this.getTime() - startDate));
            }

            let response: ts.server.protocol.CompletionEntry[] = result.response;

            let ideCompletions = this.getDetailedCompletionEx(args, response);

            if (isLogEnabled) {
                serverLogger("Completion with detailed items time, mills: " + (this.getTime() - startDate));
            }

            return {
                response: ideCompletions,
                responseRequired: true
            };
        }


        executeCommand(request: ts.server.protocol.Request): Response {
            if (request.command == TypeScriptCommandNames.IDEComposite) {
                let responses: any = {};
                let args = (request as IDECompositeRequest).arguments;
                if (!args) {
                    return doneRequest;
                }

                for (let el of args.nestedRequests) {
                    let response = this.executeCommand(el);
                    responses[el.command] = {body: response.response};
                }

                return {responseRequired: true, response: responses};
            }
            return super.executeCommand(request);
        }

        private getDetailedCompletionEx(req: ts.server.protocol.CompletionsRequestArgs,
                                        entries: ts.server.protocol.CompletionEntry[]) {
            if (!entries) {
                return entries;
            }
            const file = ts_impl.normalizePath(req.file);
            const project: ts.server.Project = this.getProjectForFileEx(file);
            if (!project) {
                serverLogger("Can't find project: shouldn't be happened", true)
                return entries;
            }
            const position = this.lineOffsetToPosition(project, file, req.line, req.offset);
            if (position == undefined) {
                return entries;
            }

            let count = 0;
            let time = this.getTime();
            return entries.reduce((accum: ts.server.protocol.CompletionEntry[], entry: ts.server.protocol.CompletionEntry) => {
                if (count <= DETAILED_COMPLETION_COUNT && ((this.getTime() - time) > DETAILED_MAX_TIME)) {
                    //no time
                    count = DETAILED_COMPLETION_COUNT + 1;
                }

                if (!isFunctionKind(entry.kind) || count++ > DETAILED_COMPLETION_COUNT) {
                    accum.push(entry);
                } else {

                    let languageService: ts.LanguageService = this.getLanguageService(project);
                    const details: any = languageService.getCompletionEntryDetails(file, position, entry.name);
                    if (details) {
                        details.sortText = entry.sortText;
                        accum.push(details);
                    }
                }
                return accum;
            }, []);
        }

        /**
         * Possible we can remove the implementation if we will use 'pull' events
         * now just for test we use 'blocking' implementation
         * to check speed of processing
         * todo use 'pull' implementation
         */
        getDiagnosticsEx(fileNames: string[], commonProject?: ts.server.Project, reqOpen: boolean = true): ts.server.protocol.DiagnosticEventBody[] {
            let projectsToProcess: {[p: string]: ts.server.Project} = {};
            let hasEmptyProject = false;
            if (commonProject) {
                let configFileName = this.getProjectName(commonProject);
                if (configFileName) {
                    projectsToProcess[configFileName] = commonProject;
                } else {
                    hasEmptyProject = true;
                }
            }


            const checkList = fileNames.reduce((accumulator: ts.server.PendingErrorCheck[], fileName: string) => {
                fileName = ts_impl.normalizePath(fileName);
                if (commonProject) {
                    accumulator.push({fileName, project: commonProject});
                } else {
                    const project: ts.server.Project = this.getProjectForFileEx(fileName);
                    if (project) {
                        accumulator.push({fileName, project});
                        let projectFilename = this.getProjectName(project);
                        if (projectFilename) {
                            projectsToProcess[projectFilename] = project;
                        } else {
                            hasEmptyProject = true;
                        }
                    }
                }
                return accumulator;
            }, []);

            let result: ts.server.protocol.DiagnosticEventBody[] = [];

            if (checkList.length > 0) {
                for (let checkSpec of checkList) {
                    let file: string = checkSpec.fileName;
                    let project = checkSpec.project;

                    if (this.containsFileEx(project, file, reqOpen)) {
                        let diagnostics: ts.server.protocol.Diagnostic[] = [];
                        const syntacticDiagnostics: ts.Diagnostic[] = this.getLanguageService(project).getSyntacticDiagnostics(file);
                        if (syntacticDiagnostics && syntacticDiagnostics.length > 0) {
                            const bakedDiagnostics = syntacticDiagnostics.map((el: ts.Diagnostic) => this.formatDiagnostic(file, checkSpec.project, el));
                            diagnostics = diagnostics.concat(bakedDiagnostics);
                        }

                        const semanticDiagnostics = this.appendPluginDiagnostics(project, this.getLanguageService(project).getSemanticDiagnostics(file), file);
                        if (semanticDiagnostics && semanticDiagnostics.length > 0) {
                            const bakedSemanticDiagnostics = semanticDiagnostics.map((el: ts.Diagnostic) => this.formatDiagnostic(file, checkSpec.project, el));
                            diagnostics = diagnostics.concat(bakedSemanticDiagnostics);

                        }

                        if (commonDefaultOptions === null && project && !this.getProjectConfigPathEx(project)) {
                            diagnostics.push({
                                start: null,
                                category: "warning",
                                end: null,
                                text: "Cannot find parent tsconfig.json"
                            });
                        }


                        if (diagnostics && diagnostics.length > 0) {
                            result.push({
                                file,
                                diagnostics
                            })
                        }
                    }

                }
            }

            result = this.appendGlobalErrors(result, projectsToProcess, hasEmptyProject);

            return result;
        }

        getForceProject(fileName: string): ts.server.Project {
            return this.getProjectForFileEx(fileName);
        }

        getMainFileDiagnosticsForFileEx(fileName: string): ts.server.protocol.DiagnosticEventBody[] {
            if (mainFile == null) {
                return this.getDiagnosticsEx([fileName]);
            }

            fileName = ts_impl.normalizePath(fileName);
            let project = this.getProjectForFileEx(mainFile);
            if (!project) {
                return [];
            }
            let resultDiagnostics = this.getDiagnosticsEx(project.getFileNames(), project, false);

            if (!this.containsFileEx(project, fileName, false)) {
                if (resultDiagnostics == null) {
                    resultDiagnostics = [];
                }
                resultDiagnostics.push({
                    file: fileName,
                    diagnostics: [{
                        start: null,
                        end: null,
                        text: "File was not processed because there is no a reference from main file"
                    }]
                });
            }

            return resultDiagnostics;
        }

        getProjectDiagnosticsForFileEx(fileName: string): ts.server.protocol.DiagnosticEventBody[] {
            let project = this.getProjectForFileEx(fileName);

            return this.getProjectDiagnosticsEx(project);
        }

        appendPluginProjectDiagnostics(project: ts.server.Project, program: ts.Program, diags: ts.server.protocol.DiagnosticEventBody[]) {
            return diags;
        }

        appendPluginDiagnostics(project: ts.server.Project, diags: ts.Diagnostic[], normalizedFileName: string) {
            return diags;
        }

        /**
         * copy formatDiag method (but we use 'TS' prefix)
         */
        formatDiagnostic(fileName: string | null, project: ts.server.Project, diagnostic: ts.Diagnostic): ts.server.protocol.Diagnostic {
            let errorText = (diagnostic.code > 0 ? ("TS" + diagnostic.code + ":") : "") + ts_impl.flattenDiagnosticMessageText(diagnostic.messageText, "\n");
            let startPosition = fileName == null ? null : this.positionToLineOffset(project, fileName, diagnostic.start);
            let endPosition = fileName == null ? null : this.positionToLineOffset(project, fileName, diagnostic.start + diagnostic.length);
            let category = diagnostic.category === ts_impl.DiagnosticCategory.Warning ? "warning" : undefined;
            return {
                start: startPosition,
                end: endPosition,
                text: errorText,
                category
            };
        }

        appendEmitDiagnostics(project: ts.server.Project, emitResult: ts.EmitResult, diagnostics: ts.server.protocol.DiagnosticEventBody[]) {
            if (diagnostics !== undefined && emitResult && emitResult.diagnostics) {
                let emitDiagnostics = emitResult.diagnostics;
                return diagnostics.concat(emitDiagnostics.map((el) => {
                        let file = el.file;
                        let fileName = file == null ? null : file.fileName;
                        let titleFile = fileName;
                        if (titleFile == null && project) {
                            titleFile = this.getProjectName(project);
                        }
                        return {file: titleFile, diagnostics: [this.formatDiagnostic(fileName, project, el)]}
                    }
                ));
            }

            return diagnostics;
        }

        getProjectDiagnosticsEx(project: ts.server.Project): ts.server.protocol.DiagnosticEventBody[] {
            if (!project) {
                return []
            }

            let program: ts.Program = this.getLanguageService(project).getProgram();
            let diagnostics: ts.Diagnostic[] = [];
            let syntax = program.getSyntacticDiagnostics();
            if (syntax && syntax.length > 0) {
                diagnostics = diagnostics.concat(syntax);
            }

            let global = program.getGlobalDiagnostics();
            if (global && global.length > 0) {
                diagnostics = diagnostics.concat(global);
            }
            let semantic = program.getSemanticDiagnostics();
            if (semantic && semantic.length > 0) {
                diagnostics = diagnostics.concat(semantic);
            }

            if (ts_impl.sortAndDeduplicateDiagnostics) {
                diagnostics = ts_impl.sortAndDeduplicateDiagnostics(diagnostics);
            }

            let fileToDiagnostics: {[p: string]: ts.server.protocol.Diagnostic[]} = {};
            let result: ts.server.protocol.DiagnosticEventBody[] = [];
            for (let diagnostic of diagnostics) {
                let sourceFile = diagnostic.file;
                if (!sourceFile) {
                    result.push({
                        file: this.getProjectConfigPathEx(project),
                        diagnostics: [this.formatDiagnostic(undefined, project, diagnostic)]
                    })
                    continue;
                }

                let fileName = ts_impl.normalizePath(sourceFile.fileName);
                let fileDiagnostics = fileToDiagnostics[fileName];
                if (!fileDiagnostics) {
                    fileDiagnostics = [];
                    fileToDiagnostics[fileName] = fileDiagnostics;
                }
                fileDiagnostics.push(this.formatDiagnostic(fileName, project, diagnostic));
            }

            if (diagnostics && diagnostics.length > 0) {
                for (let fileName in fileToDiagnostics) {
                    if (fileToDiagnostics.hasOwnProperty(fileName)) {
                        let resultDiagnostic = fileToDiagnostics[fileName];
                        if (resultDiagnostic) {
                            result.push({
                                file: fileName,
                                diagnostics: resultDiagnostic
                            });
                        }
                    }
                }
            }

            let projectsToProcess: {[p: string]: ts.server.Project} = {};
            let hasEmptyProject = false;
            let projectName = this.getProjectName(project);

            if (projectName) {
                projectsToProcess[projectName] = project;
            } else {
                hasEmptyProject = true;
            }

            result = this.appendPluginProjectDiagnostics(project, program, result);
            result = this.appendGlobalErrors(result, projectsToProcess, hasEmptyProject);

            return result;
        }
    }


    return IDESession;


    function getFileWrite(projectFilename: string, outFiles: string[], contentRoot?: string, sourceRoot?: string) {
        return (fileName: string, data?: any, writeByteOrderMark?: any, onError?: any, sourceFiles?: any[]) => {
            let normalizedName = normalizePathIfNeed(ts_impl.normalizePath(fileName), projectFilename);
            normalizedName = fixNameWithProcessor(normalizedName, onError, contentRoot, sourceRoot);
            ensureDirectoriesExist(ts_impl.getDirectoryPath(normalizedName));
            if (isLogEnabled) {
                serverLogger("Compile write file: " + fileName);
                serverLogger("Compile write file (normalized): " + normalizedName);
            }
            (<any>host).writeFile(normalizedName, data, writeByteOrderMark, onError, sourceFiles);
            outFiles.push(normalizedName);
        }
    }

    function normalizePathIfNeed(file: string, projectFilename?: string) {
        if (0 === ts_impl.getRootLength(file)) {
            let contextDir: string;
            if (projectFilename) {
                contextDir = ts_impl.getDirectoryPath(projectFilename);
            }

            if (!contextDir) {
                contextDir = host.getCurrentDirectory();
            }

            return ts_impl.getNormalizedAbsolutePath(file, contextDir);
        }

        return file;
    }

    function ensureDirectoriesExist(directoryPath: string) {
        if (directoryPath.length > ts_impl.getRootLength(directoryPath) && !host.directoryExists(directoryPath)) {
            const parentDirectory = ts_impl.getDirectoryPath(directoryPath);
            ensureDirectoriesExist(parentDirectory);
            host.createDirectory(directoryPath);
        }
    }


    function fixNameWithProcessor(filename: string, onError?: any, contentRoot?: string, sourceRoot?: string): string {
        if (pathProcessor) {
            filename = pathProcessor.getExpandedPath(filename, contentRoot, sourceRoot, onError);
        }
        return filename;
    }
}






