/*
 * Copyright 2000-2016 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.database.util;

import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.progress.ProcessCanceledException;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.util.ExceptionUtil;
import com.intellij.util.ObjectUtils;
import gnu.trove.THashSet;
import org.jetbrains.annotations.Nullable;

import java.net.UnknownHostException;
import java.util.Set;

/**
 * @author gregsh
 */
public class ErrorHandler {

  public final static Logger LOG = Logger.getInstance("com.intellij.database.util.ErrorHandler");

  private final Set<String> myMessages = new THashSet<>();
  private final StringBuilder myErrors = new StringBuilder();
  private boolean myTruncated;
  private int mySkippedErrors;

  public boolean hasErrors() {
    return myErrors.length() > 0;
  }

  public boolean addError(@Nullable String message, @Nullable Throwable ex) {
    if (ex instanceof ProcessCanceledException) throw (ProcessCanceledException)ex;
    if (myTruncated) {
      mySkippedErrors++;
      return true;
    }
    String exMessage;
    if (ex != null) {
      String m = ObjectUtils.chooseNotNull(ex.getLocalizedMessage(), ex.getMessage());
      m = StringUtil.isNotEmpty(m) ? m : ex.getClass().getName();
      exMessage = m.equals(message) ? null : m;
    }
    else {
      exMessage = null;
    }

    // do not log similar errors
    if (message != null && !myMessages.add(message) ||
        exMessage != null && !myMessages.add(exMessage)) {
      mySkippedErrors++;
      return true;
    }
    if (ex != null) {
      ExceptionUtil.getUserStackTrace(ex, LOG);
      LOG.warn(ex);
    }

    if (message != null) myErrors.append(message).append(".\n");
    if (exMessage != null) {
      Throwable cause = ex.getCause();
      if (cause instanceof UnknownHostException) {
        myErrors.append("Host ").append(cause.getMessage()).append(" is unknown.\n");
      }
      myErrors.append(StringUtil.replace(exMessage, "\n\n", ".\n")).append("\n");
    }

    if (myErrors.length() > 10240) myTruncated = true;
    return true;
  }

  public String getSummary() {
    if (mySkippedErrors > 0) {
      myErrors.append("and ").append(mySkippedErrors);
      if (myTruncated) {
        myErrors.append(" more");
      }
      else {
        myErrors.append(" duplicate reports");
      }
      mySkippedErrors = 0;
    }
    return myErrors.toString();
  }

  public void setCaption(String message) {
    myErrors.insert(0, message + "\n");
  }
}
